<?php
require_once '../db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get request body
$data = json_decode(file_get_contents('php://input'), true);

// Check if required fields are provided
if (!isset($data['user_id']) || !isset($data['current_pin']) || !isset($data['new_pin'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'User ID, current PIN, and new PIN are required']);
    exit;
}

$userId = $data['user_id'];
$currentPin = $data['current_pin'];
$newPin = $data['new_pin'];

// Validate PIN format (4 digits)
if (!preg_match('/^\d{4}$/', $newPin)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'New PIN must be 4 digits']);
    exit;
}

try {
    // Check if user exists and current PIN is correct
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? LIMIT 1");
    $stmt->execute([$userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'User not found']);
        exit;
    }

    // Verify current PIN (in a real app, this would be hashed)
    if ($user['password'] !== $currentPin) {
        // Log failed PIN change attempt
        $stmt = $pdo->prepare("INSERT INTO activity_logs (user_id, activity_type, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$userId, 'pin_change_failed', "Failed PIN change attempt for user ID: $userId (incorrect current PIN)", $_SERVER['REMOTE_ADDR']]);

        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Current PIN is incorrect']);
        exit;
    }

    // Update PIN
    $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
    $stmt->execute([$newPin, $userId]);

    // Log successful PIN change
    $stmt = $pdo->prepare("INSERT INTO activity_logs (user_id, activity_type, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
    $stmt->execute([$userId, 'pin_changed', "PIN changed successfully for user ID: $userId", $_SERVER['REMOTE_ADDR']]);

    // Return success response
    echo json_encode(['success' => true, 'message' => 'PIN changed successfully']);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
?>
