<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Get request body
$data = json_decode(file_get_contents('php://input'), true);

// Check if username and PIN are provided
if (!isset($data['username']) || !isset($data['pin'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Username and PIN are required']);
    exit;
}

$username = $data['username'];
$pin = $data['pin'];

// Validate PIN format (4 digits)
if (!preg_match('/^\d{4}$/', $pin)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'PIN must be 4 digits']);
    exit;
}

try {
    // Check if user exists and PIN is correct
    $stmt = $pdo->prepare("SELECT * FROM users WHERE user_identifier = ? OR email = ? LIMIT 1");
    $stmt->execute([$username, $username]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // Log failed login attempt
        logActivity($pdo, null, 'login_failed', "Failed login attempt (user not found)");
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Invalid username or PIN']);
        exit;
    }

    // Check if account is locked
    if ($user['status'] === 'locked') {
        logActivity($pdo, $user['id'], 'login_failed', "Login attempt on locked account");
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Account is locked. Please contact administrator']);
        exit;
    }

    // Check if account is inactive
    if ($user['status'] !== 'active') {
        logActivity($pdo, $user['id'], 'login_failed', "Login attempt on inactive account");
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Account is not active. Please contact administrator']);
        exit;
    }

    // Verify PIN
    if ($user['password'] !== $pin) {
        // Log failed attempt and update counter
        logActivity($pdo, $user['id'], 'login_failed', "Failed login attempt (incorrect PIN)");
        updateFailedLoginAttempts($pdo, $user['id']);
        
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Invalid username or PIN']);
        exit;
    }

    // Success - Reset failed attempts and generate token
    resetFailedLoginAttempts($pdo, $user['id']);
    $token = generateToken($user['id']);
    updateLastLogin($pdo, $user['id']);
    logActivity($pdo, $user['id'], 'login', "Successful login");

    // Prepare user data
    $userData = [
        'id' => $user['id'],
        'username' => $user['user_identifier'],
        'email' => $user['email'],
        'name' => $user['name'],
        'role' => $user['role'],
        'created_at' => $user['created_at'],
        'last_login' => $user['updated_at'],
        'biometric_enabled' => (bool)$user['biometric_enabled']
    ];

    // Return success response
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Login successful',
        'user' => $userData,
        'token' => $token
    ]);

} catch (PDOException $e) {
    error_log("Login error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Server error occurred']);
}

// Helper functions

function generateToken($userId) {
    // In a real app, use a proper JWT library
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode([
        'user_id' => $userId,
        'iat' => time(),
        'exp' => time() + (60 * 60 * 24) // 24 hours
    ]);

    $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));

    $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, 'your_secret_key', true);
    $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));

    return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
}

function updateLastLogin($pdo, $userId) {
    $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->execute([$userId]);
}

function updateFailedLoginAttempts($pdo, $userId) {
    $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = failed_login_attempts + 1 WHERE id = ?");
    $stmt->execute([$userId]);

    // Check if account should be locked (after 5 failed attempts)
    $stmt = $pdo->prepare("SELECT failed_login_attempts FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $attempts = $stmt->fetchColumn();

    if ($attempts >= 5) {
        $stmt = $pdo->prepare("UPDATE users SET status = 'locked' WHERE id = ?");
        $stmt->execute([$userId]);
    }
}

function resetFailedLoginAttempts($pdo, $userId) {
    $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = 0 WHERE id = ?");
    $stmt->execute([$userId]);
}

function logActivity($pdo, $userId, $activityType, $details) {
    try {
        $stmt = $pdo->prepare("INSERT INTO activity_logs (user_id, activity_type, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$userId, $activityType, $details, $_SERVER['REMOTE_ADDR']]);
    } catch (PDOException $e) {
        // Silently fail - don't let logging errors affect the main functionality
        error_log("Error logging activity: " . $e->getMessage());
    }
}
?>

