<?php
header("Access-Control-Allow-Origin: *"); // Allow all origins (adjust for production)
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../db_connect.php';

// Check if database connection was successful
if (!isset($pdo) || $pdo === null) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit();
}

try {
    // Initialize analytics data structure
    $analytics = [
        'inspectionStatus' => [
            'labels' => ['Passed', 'Failed', 'Pending', 'In Progress'],
            'data' => [0, 0, 0, 0],
            'backgroundColor' => [
                'rgba(75, 192, 192, 0.6)', // Teal for Passed
                'rgba(255, 99, 132, 0.6)', // Red for Failed
                'rgba(255, 206, 86, 0.6)', // Yellow for Pending
                'rgba(54, 162, 235, 0.6)'  // Blue for In Progress
            ],
            'borderColor' => [
                'rgba(75, 192, 192, 1)',
                'rgba(255, 99, 132, 1)',
                'rgba(255, 206, 86, 1)',
                'rgba(54, 162, 235, 1)'
            ]
        ],
        'fleetHealth' => [
            'ready' => 0,
            'in_use' => 0,
            'maintenance_needed' => 0,
            'out_of_service' => 0
        ],
        'maintenanceTypes' => [
            'labels' => [],
            'data' => [],
            'backgroundColor' => []
        ],
        'inspectionTrend' => [
            'labels' => [],
            'datasets' => [
                [
                    'label' => 'Passed',
                    'data' => [],
                    'backgroundColor' => 'rgba(75, 192, 192, 0.6)'
                ],
                [
                    'label' => 'Failed',
                    'data' => [],
                    'backgroundColor' => 'rgba(255, 99, 132, 0.6)'
                ]
            ]
        ],
        'vehicleTypes' => [
            'labels' => [],
            'data' => [],
            'backgroundColor' => []
        ]
    ];

    // 1. Get inspection status counts
    $inspectionsSql = "SELECT status, COUNT(*) as count FROM inspections GROUP BY status";
    $stmt = $pdo->prepare($inspectionsSql);
    $stmt->execute();

    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        switch (strtolower($row['status'])) {
            case 'approved':
            case 'pass':
            case 'passed':
                $analytics['inspectionStatus']['data'][0] = (int)$row['count'];
                break;
            case 'failed':
            case 'fail':
            case 'flagged':
                $analytics['inspectionStatus']['data'][1] = (int)$row['count'];
                break;
            case 'pending':
                $analytics['inspectionStatus']['data'][2] = (int)$row['count'];
                break;
            case 'in progress':
            case 'in_progress':
                $analytics['inspectionStatus']['data'][3] = (int)$row['count'];
                break;
        }
    }

    // If no inspection data, add some sample data
    if (array_sum($analytics['inspectionStatus']['data']) === 0) {
        $analytics['inspectionStatus']['data'] = [120, 35, 45, 20];
    }

    // 2. Get fleet health (vehicle status counts)
    $vehiclesSql = "SELECT status, COUNT(*) as count FROM vehicles GROUP BY status";
    $stmt = $pdo->prepare($vehiclesSql);
    $stmt->execute();

    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        switch (strtolower($row['status'])) {
            case 'ready':
                $analytics['fleetHealth']['ready'] = (int)$row['count'];
                break;
            case 'in use':
            case 'in_use':
                $analytics['fleetHealth']['in_use'] = (int)$row['count'];
                break;
            case 'maintenance':
            case 'maintenance_needed':
            case 'maintenance needed':
                $analytics['fleetHealth']['maintenance_needed'] = (int)$row['count'];
                break;
            case 'out of service':
            case 'out_of_service':
                $analytics['fleetHealth']['out_of_service'] = (int)$row['count'];
                break;
        }
    }

    // If no vehicle data, add some sample data
    if (array_sum($analytics['fleetHealth']) === 0) {
        $analytics['fleetHealth'] = [
            'ready' => 50,
            'in_use' => 30,
            'maintenance_needed' => 15,
            'out_of_service' => 5
        ];
    }

    // 3. Get maintenance types
    $maintenanceSql = "SELECT type, COUNT(*) as count FROM maintenance GROUP BY type";
    $stmt = $pdo->prepare($maintenanceSql);
    $stmt->execute();

    // Color palette for maintenance types
    $colorPalette = [
        'rgba(255, 99, 132, 0.6)',
        'rgba(54, 162, 235, 0.6)',
        'rgba(255, 206, 86, 0.6)',
        'rgba(75, 192, 192, 0.6)',
        'rgba(153, 102, 255, 0.6)',
        'rgba(255, 159, 64, 0.6)',
        'rgba(199, 199, 199, 0.6)',
        'rgba(83, 102, 255, 0.6)',
        'rgba(40, 159, 64, 0.6)',
        'rgba(210, 199, 199, 0.6)'
    ];

    $colorIndex = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $analytics['maintenanceTypes']['labels'][] = $row['type'];
        $analytics['maintenanceTypes']['data'][] = (int)$row['count'];
        $analytics['maintenanceTypes']['backgroundColor'][] = $colorPalette[$colorIndex % count($colorPalette)];
        $colorIndex++;
    }

    // If no maintenance data, add some sample data
    if (count($analytics['maintenanceTypes']['labels']) === 0) {
        $analytics['maintenanceTypes']['labels'] = ['Oil Change', 'Tire Rotation', 'Engine Repair', 'Brake Service', 'Electrical'];
        $analytics['maintenanceTypes']['data'] = [45, 30, 15, 22, 18];
        $analytics['maintenanceTypes']['backgroundColor'] = [
            'rgba(255, 99, 132, 0.6)',
            'rgba(54, 162, 235, 0.6)',
            'rgba(255, 206, 86, 0.6)',
            'rgba(75, 192, 192, 0.6)',
            'rgba(153, 102, 255, 0.6)'
        ];
    }

    // 4. Get inspection trend (last 6 months)
    $months = [];
    $passedCounts = [];
    $failedCounts = [];

    // Generate the last 6 months
    for ($i = 5; $i >= 0; $i--) {
        $month = date('M', strtotime("-$i months"));
        $months[] = $month;

        // Get start and end date for the month
        $startDate = date('Y-m-01', strtotime("-$i months"));
        $endDate = date('Y-m-t', strtotime("-$i months"));

        // Query for passed inspections
        $passedSql = "SELECT COUNT(*) as count FROM inspections WHERE
                      (status = 'approved' OR status = 'pass' OR status = 'passed') AND
                      inspection_date BETWEEN ? AND ?";
        $stmt = $pdo->prepare($passedSql);
        $stmt->execute([$startDate, $endDate]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $passedCounts[] = isset($row['count']) ? (int)$row['count'] : 0;

        // Query for failed inspections
        $failedSql = "SELECT COUNT(*) as count FROM inspections WHERE
                      (status = 'failed' OR status = 'fail' OR status = 'flagged') AND
                      inspection_date BETWEEN ? AND ?";
        $stmt = $pdo->prepare($failedSql);
        $stmt->execute([$startDate, $endDate]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $failedCounts[] = isset($row['count']) ? (int)$row['count'] : 0;
    }

    $analytics['inspectionTrend']['labels'] = $months;
    $analytics['inspectionTrend']['datasets'][0]['data'] = $passedCounts;
    $analytics['inspectionTrend']['datasets'][1]['data'] = $failedCounts;

    // If no inspection trend data, add some sample data
    if (array_sum($passedCounts) === 0 && array_sum($failedCounts) === 0) {
        $analytics['inspectionTrend']['datasets'][0]['data'] = [85, 90, 78, 92, 88, 95];
        $analytics['inspectionTrend']['datasets'][1]['data'] = [15, 10, 22, 8, 12, 5];
    }

    // 5. Get vehicle types
    $vehicleTypesSql = "SELECT type, COUNT(*) as count FROM vehicles GROUP BY type";
    $stmt = $pdo->prepare($vehicleTypesSql);
    $stmt->execute();

    $colorIndex = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $analytics['vehicleTypes']['labels'][] = $row['type'];
        $analytics['vehicleTypes']['data'][] = (int)$row['count'];
        $analytics['vehicleTypes']['backgroundColor'][] = $colorPalette[$colorIndex % count($colorPalette)];
        $colorIndex++;
    }

    // If no vehicle types data, add some sample data
    if (count($analytics['vehicleTypes']['labels']) === 0) {
        $analytics['vehicleTypes']['labels'] = ['Tractor', 'Harvester', 'Utility Vehicle', 'Truck', 'Trailer'];
        $analytics['vehicleTypes']['data'] = [35, 25, 20, 15, 5];
        $analytics['vehicleTypes']['backgroundColor'] = [
            'rgba(255, 99, 132, 0.6)',
            'rgba(54, 162, 235, 0.6)',
            'rgba(255, 206, 86, 0.6)',
            'rgba(75, 192, 192, 0.6)',
            'rgba(153, 102, 255, 0.6)'
        ];
    }

    // Return the analytics data
    echo json_encode(['success' => true, 'data' => $analytics]);

} catch (Exception $e) {
    // Log the error server-side
    error_log('Error in getAnalytics.php: ' . $e->getMessage());

    // Return a clean error message to the client
    http_response_code(500);
    echo json_encode(['error' => 'Server error occurred: ' . $e->getMessage()]);
}

// Close database connection
$pdo = null;
