<?php
require_once '../db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// --- Authentication Placeholder ---
// In a real app, verify the user's token/session and get their ID and role
// $user_id = $_SESSION['user_id'] ?? null;
// $user_role = $_SESSION['user_role'] ?? null;
$user_id = 7; // Hardcoding Dispatcher One (ID 7) for now, replace with actual auth
$user_role = 'dispatch';
// --- End Authentication Placeholder ---

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['error' => 'Method not allowed.']);
    exit;
}

try {
    // Get all dashboard data in a single API call
    $response = [
        'fleetStatus' => getFleetStatus($pdo),
        'tasks' => getTasks($pdo),
        'alerts' => getAlerts($pdo),
        'inspectionStats' => getInspectionStats($pdo),
        'upcomingMaintenance' => getUpcomingMaintenance($pdo),
        'userProfile' => getUserProfile($pdo, $user_id),
        'recentInspections' => getRecentInspections($pdo),
        'quickStats' => getQuickStats($pdo)
    ];

    echo json_encode($response);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}

// Function to get fleet status
function getFleetStatus($pdo) {
    try {
        // Query to count vehicles by status
        $sql = "SELECT
                    SUM(CASE WHEN status = 'available' THEN 1 ELSE 0 END) as available,
                    SUM(CASE WHEN status = 'in_use' THEN 1 ELSE 0 END) as inUse,
                    SUM(CASE WHEN status = 'maintenance' THEN 1 ELSE 0 END) as maintenance,
                    SUM(CASE WHEN status = 'out_of_service' THEN 1 ELSE 0 END) as outOfService
                FROM vehicles";

        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        // Convert string values to integers
        return [
            'available' => (int)($result['available'] ?? 0),
            'inUse' => (int)($result['inUse'] ?? 0),
            'maintenance' => (int)($result['maintenance'] ?? 0),
            'outOfService' => (int)($result['outOfService'] ?? 0)
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'available' => 18,
            'inUse' => 42,
            'maintenance' => 7,
            'outOfService' => 3
        ];
    }
}

// Function to get pending tasks
function getTasks($pdo) {
    try {
        $sql = "SELECT
                    t.id,
                    t.title,
                    t.description,
                    t.priority,
                    t.due_date,
                    t.status,
                    t.assignee_id,
                    u.name as assignee
                FROM tasks t
                LEFT JOIN users u ON t.assignee_id = u.id
                WHERE t.status = 'pending'
                ORDER BY
                    CASE
                        WHEN t.priority = 'High' THEN 1
                        WHEN t.priority = 'Medium' THEN 2
                        ELSE 3
                    END,
                    t.due_date ASC";

        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Log the tasks for debugging
        error_log('Fetched tasks: ' . print_r($tasks, true));

        return $tasks;
    } catch (PDOException $e) {
        // Log the error
        error_log('Error fetching tasks: ' . $e->getMessage());
        // Return empty array if query fails
        return [];
    }
}

// Function to get alerts
function getAlerts($pdo) {
    try {
        $sql = "SELECT
                    id,
                    type,
                    message,
                    created_at as time,
                    is_read
                FROM alerts
                WHERE is_read = 0
                ORDER BY created_at DESC
                LIMIT 5";

        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Return empty array if query fails
        return [];
    }
}

// Function to get inspection statistics
function getInspectionStats($pdo) {
    try {
        $sql = "SELECT
                    SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                    SUM(CASE WHEN status = 'scheduled' THEN 1 ELSE 0 END) as scheduled
                FROM inspections";

        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        // Convert string values to integers
        return [
            'completed' => (int)($result['completed'] ?? 0),
            'pending' => (int)($result['pending'] ?? 0),
            'failed' => (int)($result['failed'] ?? 0),
            'scheduled' => (int)($result['scheduled'] ?? 0)
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'completed' => 78,
            'pending' => 23,
            'failed' => 5,
            'scheduled' => 31
        ];
    }
}

// Function to get upcoming maintenance
function getUpcomingMaintenance($pdo) {
    try {
        $sql = "SELECT
                    m.id,
                    CONCAT(v.type, ' #', v.identifier) as vehicle,
                    m.scheduled_date as date,
                    m.maintenance_type as type
                FROM maintenance m
                JOIN vehicles v ON m.vehicle_id = v.id
                WHERE m.status = 'scheduled' AND m.scheduled_date >= CURDATE()
                ORDER BY m.scheduled_date ASC
                LIMIT 4";

        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            ['id' => 1, 'vehicle' => 'Truck #142', 'date' => '2025-03-22', 'type' => 'Oil Change'],
            ['id' => 2, 'vehicle' => 'Van #087', 'date' => '2025-03-23', 'type' => 'Brake Inspection'],
            ['id' => 3, 'vehicle' => 'Truck #265', 'date' => '2025-03-24', 'type' => 'Tire Rotation'],
            ['id' => 4, 'vehicle' => 'SUV #112', 'date' => '2025-03-25', 'type' => 'Full Service']
        ];
    }
}

// Function to get user profile
function getUserProfile($pdo, $user_id) {
    try {
        $sql = "SELECT
                    id,
                    name,
                    role,
                    email,
                    last_login
                FROM users
                WHERE id = :user_id";

        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->execute();

        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($user) {
            // Calculate time since last login
            $lastActive = 'Just now';
            if (!empty($user['last_login'])) {
                $lastLogin = new DateTime($user['last_login']);
                $now = new DateTime();
                $diff = $now->diff($lastLogin);

                if ($diff->days > 0) {
                    $lastActive = $diff->days . ' days ago';
                } elseif ($diff->h > 0) {
                    $lastActive = $diff->h . ' hours ago';
                } elseif ($diff->i > 0) {
                    $lastActive = $diff->i . ' minutes ago';
                }
            }

            return [
                'name' => $user['name'],
                'role' => ucfirst($user['role']) . ' Manager',
                'avatar' => 'https://via.placeholder.com/80', // Placeholder for now
                'status' => 'Active',
                'lastActive' => $lastActive
            ];
        }

        // Return default if user not found
        return [
            'name' => 'John Doe',
            'role' => 'Dispatch Manager',
            'avatar' => 'https://via.placeholder.com/80',
            'status' => 'Active',
            'lastActive' => '2 minutes ago'
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'name' => 'John Doe',
            'role' => 'Dispatch Manager',
            'avatar' => 'https://via.placeholder.com/80',
            'status' => 'Active',
            'lastActive' => '2 minutes ago'
        ];
    }
}

// Function to get recent inspections
function getRecentInspections($pdo) {
    try {
        $sql = "SELECT
                    i.id,
                    CONCAT(v.type, ' #', v.identifier) as vehicle,
                    u.name as inspector,
                    i.inspection_date as date,
                    i.status as result
                FROM inspections i
                JOIN vehicles v ON i.vehicle_id = v.id
                JOIN users u ON i.inspector_id = u.id
                ORDER BY i.inspection_date DESC
                LIMIT 3";

        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            ['id' => 1, 'vehicle' => 'Truck #156', 'inspector' => 'Alex Johnson', 'date' => '2025-03-19', 'result' => 'passed'],
            ['id' => 2, 'vehicle' => 'Van #092', 'inspector' => 'Sarah Miller', 'date' => '2025-03-19', 'result' => 'failed'],
            ['id' => 3, 'vehicle' => 'Truck #238', 'inspector' => 'David Wilson', 'date' => '2025-03-18', 'result' => 'passed']
        ];
    }
}

// Function to get quick stats
function getQuickStats($pdo) {
    try {
        // Get active drivers count
        $stmt1 = $pdo->prepare("SELECT COUNT(*) as active_drivers FROM users WHERE role = 'driver' AND status = 'active'");
        $stmt1->execute();
        $activeDrivers = $stmt1->fetch(PDO::FETCH_ASSOC)['active_drivers'] ?? 0;

        // Get pending inspections count
        $stmt2 = $pdo->prepare("SELECT COUNT(*) as pending_inspections FROM inspections WHERE status = 'pending'");
        $stmt2->execute();
        $pendingInspections = $stmt2->fetch(PDO::FETCH_ASSOC)['pending_inspections'] ?? 0;

        // Get completed inspections today
        $stmt3 = $pdo->prepare("SELECT COUNT(*) as completed_today FROM inspections WHERE status = 'completed' AND DATE(inspection_date) = CURDATE()");
        $stmt3->execute();
        $completedToday = $stmt3->fetch(PDO::FETCH_ASSOC)['completed_today'] ?? 0;

        // Get average inspection time
        $stmt4 = $pdo->prepare("SELECT AVG(TIMESTAMPDIFF(MINUTE, start_time, end_time)) as avg_time FROM inspections WHERE status = 'completed' AND start_time IS NOT NULL AND end_time IS NOT NULL");
        $stmt4->execute();
        $avgTime = round($stmt4->fetch(PDO::FETCH_ASSOC)['avg_time'] ?? 0);

        return [
            'activeDrivers' => (int)$activeDrivers,
            'pendingInspections' => (int)$pendingInspections,
            'completedToday' => (int)$completedToday,
            'avgInspectionTime' => $avgTime . ' min'
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'activeDrivers' => 12,
            'pendingInspections' => 5,
            'completedToday' => 8,
            'avgInspectionTime' => '24 min'
        ];
    }
}
?>
