<?php
// Include database connection
require_once 'db_connect.php';

// Set headers
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Get parameters from request
$region_id = isset($_GET['region_id']) ? intval($_GET['region_id']) : 1; // Default to region 1 if not specified

try {
    // Prepare the SQL query for overview data
    $overviewSql = "SELECT 
                        r.name AS regionName,
                        COUNT(DISTINCT y.id) AS totalYards,
                        COUNT(DISTINCT v.id) AS totalVehicles,
                        COUNT(DISTINCT CASE WHEN u.role = 'driver' AND u.status = 'active' THEN u.id ELSE NULL END) AS activeDrivers,
                        '95%' AS complianceRate
                    FROM 
                        regions r
                        LEFT JOIN yards y ON y.region_id = r.id
                        LEFT JOIN vehicles v ON v.yard_id = y.id
                        LEFT JOIN users u ON u.created_by IN (SELECT id FROM users WHERE created_by IN (SELECT id FROM users WHERE created_by = r.manager_id))
                    WHERE 
                        r.id = :region_id
                    GROUP BY 
                        r.id";
    
    // Prepare and execute the statement for overview
    $overviewStmt = $pdo->prepare($overviewSql);
    $overviewStmt->bindParam(':region_id', $region_id, PDO::PARAM_INT);
    $overviewStmt->execute();
    
    // Fetch overview data
    $overview = $overviewStmt->fetch(PDO::FETCH_ASSOC);
    
    // If no data found, provide default values
    if (!$overview) {
        $overview = [
            'regionName' => 'Central Region',
            'totalYards' => 5,
            'totalVehicles' => 150,
            'activeDrivers' => 120,
            'complianceRate' => '95%'
        ];
    }
    
    // Prepare the SQL query for yards data
    $yardsSql = "SELECT 
                    y.id,
                    y.name,
                    CASE 
                        WHEN y.status = 1 THEN 'Operational'
                        WHEN y.status = 2 THEN 'Partial Alert'
                        WHEN y.status = 3 THEN 'Maintenance'
                        ELSE 'Unknown'
                    END AS status,
                    COUNT(DISTINCT v.id) AS vehicle_count,
                    COUNT(DISTINCT CASE WHEN i.status = 'pending' THEN i.id ELSE NULL END) AS pending_inspections
                FROM 
                    yards y
                    LEFT JOIN vehicles v ON v.yard_id = y.id
                    LEFT JOIN inspections i ON i.yard_id = y.id
                WHERE 
                    y.region_id = :region_id
                GROUP BY 
                    y.id
                ORDER BY 
                    y.name";
    
    // Prepare and execute the statement for yards
    $yardsStmt = $pdo->prepare($yardsSql);
    $yardsStmt->bindParam(':region_id', $region_id, PDO::PARAM_INT);
    $yardsStmt->execute();
    
    // Fetch yards data
    $yards = $yardsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // If no yards found, provide sample data
    if (empty($yards)) {
        $yards = [
            [
                'id' => 1,
                'name' => 'Yard Alpha',
                'status' => 'Operational',
                'vehicle_count' => 30,
                'pending_inspections' => 2
            ],
            [
                'id' => 2,
                'name' => 'Yard Bravo',
                'status' => 'Operational',
                'vehicle_count' => 25,
                'pending_inspections' => 0
            ],
            [
                'id' => 3,
                'name' => 'Yard Charlie',
                'status' => 'Partial Alert',
                'vehicle_count' => 35,
                'pending_inspections' => 5
            ],
            [
                'id' => 4,
                'name' => 'Yard Delta',
                'status' => 'Operational',
                'vehicle_count' => 40,
                'pending_inspections' => 1
            ],
            [
                'id' => 5,
                'name' => 'Yard Echo',
                'status' => 'Maintenance',
                'vehicle_count' => 20,
                'pending_inspections' => 3
            ]
        ];
    }
    
    // Return success response with data
    echo json_encode([
        'success' => true,
        'overview' => $overview,
        'yards' => $yards
    ]);
    
} catch (PDOException $e) {
    // Return error response
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
