<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../db_connect.php';

// Check if database connection was successful
if (!isset($pdo) || $pdo === null) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit();
}

try {
    // Get region_id from request (optional, can be used for filtering)
    $region_id = isset($_GET['region_id']) ? intval($_GET['region_id']) : null;
    
    // Prepare the response data
    $response = [
        'summary' => [],
        'breakdown' => [],
        'issues' => []
    ];
    
    // 1. Get compliance summary data
    // This would typically involve querying inspection records, maintenance records, etc.
    // For now, we'll generate some sample data
    
    // Count total inspections
    $inspectionQuery = "SELECT 
                            COUNT(*) as total_inspections,
                            SUM(CASE WHEN status = 'passed' THEN 1 ELSE 0 END) as passed_inspections
                        FROM inspections";
    
    // Add region filter if provided
    if ($region_id) {
        // This assumes you have a way to link inspections to regions
        // You might need to join with vehicles or yards table
        $inspectionQuery .= " WHERE region_id = :region_id";
    }
    
    $stmt = $pdo->prepare($inspectionQuery);
    
    if ($region_id) {
        $stmt->bindParam(':region_id', $region_id, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    $inspectionData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $totalInspections = $inspectionData['total_inspections'] ?? 0;
    $passedInspections = $inspectionData['passed_inspections'] ?? 0;
    
    // Calculate compliance rate
    $complianceRate = $totalInspections > 0 ? round(($passedInspections / $totalInspections) * 100) : 0;
    
    // Count overdue maintenance actions
    $overdueQuery = "SELECT COUNT(*) as overdue_count FROM maintenance WHERE status = 'scheduled' AND start_date < CURDATE()";
    $stmt = $pdo->prepare($overdueQuery);
    $stmt->execute();
    $overdueData = $stmt->fetch(PDO::FETCH_ASSOC);
    $overdueCount = $overdueData['overdue_count'] ?? 0;
    
    // Populate summary data
    $response['summary'] = [
        'overallRate' => $complianceRate . '%',
        'inspectionsPassed' => $passedInspections,
        'inspectionsTotal' => $totalInspections,
        'overdueActions' => $overdueCount,
        'regulatoryChecks' => 'Up-to-date'
    ];
    
    // 2. Get compliance breakdown by area
    // This would typically involve querying different categories of compliance
    
    // Sample breakdown data - in a real app, this would come from database queries
    $response['breakdown'] = [
        [
            'area' => 'Pre-Trip Inspections',
            'rate' => '98%',
            'status' => 'Good'
        ],
        [
            'area' => 'Maintenance Schedules',
            'rate' => '92%',
            'status' => 'Needs Attention'
        ],
        [
            'area' => 'Driver Hours Logged',
            'rate' => '99%',
            'status' => 'Excellent'
        ],
        [
            'area' => 'Vehicle Registrations',
            'rate' => '100%',
            'status' => 'Excellent'
        ]
    ];
    
    // 3. Get recent compliance issues
    // Query for failed inspections or maintenance issues
    $issuesQuery = "SELECT 
                        'I' || i.id as id,
                        'Failed Inspection' as issue_type,
                        v.name as vehicle_identifier,
                        u.name as driver_name,
                        y.name as yard_name,
                        DATE_FORMAT(i.inspection_date, '%Y-%m-%d') as issue_date
                    FROM 
                        inspections i
                    LEFT JOIN 
                        vehicles v ON i.vehicle_id = v.id
                    LEFT JOIN 
                        users u ON i.inspector_id = u.id
                    LEFT JOIN 
                        yards y ON v.yard_id = y.id
                    WHERE 
                        i.status = 'failed'
                    ORDER BY 
                        i.inspection_date DESC
                    LIMIT 5";
    
    $stmt = $pdo->prepare($issuesQuery);
    $stmt->execute();
    $issues = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Add maintenance issues
    $maintenanceIssuesQuery = "SELECT 
                                'M' || m.id as id,
                                CONCAT(m.type, ' Needed') as issue_type,
                                v.name as vehicle_identifier,
                                NULL as driver_name,
                                y.name as yard_name,
                                DATE_FORMAT(m.start_date, '%Y-%m-%d') as issue_date
                            FROM 
                                maintenance m
                            LEFT JOIN 
                                vehicles v ON m.vehicle_id = v.id
                            LEFT JOIN 
                                yards y ON v.yard_id = y.id
                            WHERE 
                                m.status = 'scheduled' AND m.start_date < CURDATE()
                            ORDER BY 
                                m.start_date ASC
                            LIMIT 5";
    
    $stmt = $pdo->prepare($maintenanceIssuesQuery);
    $stmt->execute();
    $maintenanceIssues = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Combine and sort issues
    $allIssues = array_merge($issues, $maintenanceIssues);
    usort($allIssues, function($a, $b) {
        return strtotime($b['issue_date']) - strtotime($a['issue_date']);
    });
    
    // Take only the 5 most recent issues
    $response['issues'] = array_slice($allIssues, 0, 5);
    
    // Return the response
    echo json_encode($response);
    
} catch (Exception $e) {
    // Log the error server-side
    error_log('Error in compliance.php: ' . $e->getMessage());
    
    // Return a clean error message to the client
    http_response_code(500);
    echo json_encode(['error' => 'Server error occurred: ' . $e->getMessage()]);
}

// Close database connection
$pdo = null;
?>
