<?php
header("Access-Control-Allow-Origin: *"); // Allow requests from any origin (adjust for production)
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, OPTIONS"); // Allow POST and OPTIONS methods
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request (pre-flight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../../db_connect.php'; // Adjust path as needed

// Get posted data
$data = json_decode(file_get_contents("php://input"));

// Basic validation
if (
    !isset($data->name) || empty($data->name) ||
    !isset($data->email) || empty($data->email) || !filter_var($data->email, FILTER_VALIDATE_EMAIL) ||
    !isset($data->password) || empty($data->password) || strlen($data->password) < 8 || // Require minimum password length
    !isset($data->role) || empty($data->role) ||
    !isset($data->status) || empty($data->status)
) {
    http_response_code(400); // Bad Request
    echo json_encode(array("error" => "Incomplete or invalid user data provided."));
    exit();
}

// Sanitize input (basic example)
$name = htmlspecialchars(strip_tags($data->name));
$email = filter_var($data->email, FILTER_SANITIZE_EMAIL);
$password = $data->password; // Do NOT sanitize the password directly before hashing
$role = htmlspecialchars(strip_tags($data->role));
$status = htmlspecialchars(strip_tags($data->status));

// Validate role and status against allowed enum values (important!)
$allowed_roles = ['owner', 'regional_manager', 'yard_manager', 'field_rep', 'dispatch', 'driver', 'wash_crew'];
$allowed_statuses = ['active', 'inactive', 'pending'];

if (!in_array($role, $allowed_roles)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid role specified."));
    exit();
}

if (!in_array($status, $allowed_statuses)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid status specified."));
    exit();
}

try {
    // $pdo is created in db_connect.php
    if (!isset($pdo) || !$pdo instanceof PDO) {
        // This condition might be hit if db_connect.php failed silently
        throw new Exception("Database connection object not available.");
    }

    // Check if email already exists
    $checkEmailSql = "SELECT id FROM users WHERE email = :email";
    $checkEmailStmt = $pdo->prepare($checkEmailSql);
    $checkEmailStmt->bindParam(':email', $email);
    $checkEmailStmt->execute();

    if ($checkEmailStmt->rowCount() > 0) {
        http_response_code(409); // Conflict
        echo json_encode(array("error" => "Email address is already in use."));
        exit();
    }

    // Hash the password securely
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

    // Prepare insert statement
    $sql = "INSERT INTO users (name, email, password, role, status, created_at, updated_at) VALUES (:name, :email, :password, :role, :status, NOW(), NOW())";
    $stmt = $pdo->prepare($sql);

    // Bind parameters
    $stmt->bindParam(':name', $name);
    $stmt->bindParam(':email', $email);
    $stmt->bindParam(':password', $hashedPassword);
    $stmt->bindParam(':role', $role);
    $stmt->bindParam(':status', $status);

    // Execute the query
    if ($stmt->execute()) {
        http_response_code(201); // Created
        echo json_encode(array("message" => "User registered successfully."));
    } else {
        http_response_code(503); // Service Unavailable (database error)
        echo json_encode(array("error" => "Unable to register user. Database error."));
    }

} catch (PDOException $e) {
    http_response_code(503); // Service Unavailable
    echo json_encode(array("error" => "Database connection error: " . $e->getMessage()));
} catch (Exception $e) {
    http_response_code(500); // Internal Server Error
    echo json_encode(array("error" => "An unexpected error occurred: " . $e->getMessage()));
} finally {
    $pdo = null; // Close connection
}
?>
