<?php
header("Access-Control-Allow-Origin: *"); // Allow all origins (adjust for production)
header("Access-Control-Allow-Methods: GET, POST, DELETE, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../../db_connect.php';

// Check if database connection was successful
if (!isset($pdo) || $pdo === null) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit();
}

// --- Authentication Placeholder ---
// In a real app, you'd verify the user's session/token and ensure they have owner privileges
// --- End Authentication Placeholder ---

// Wrap everything in a try-catch to ensure we always return valid JSON
try {
    // Check the structure of the reports table if it exists
    $checkTableSql = "SHOW TABLES LIKE 'reports'";
    $stmt = $pdo->prepare($checkTableSql);
    $stmt->execute();
    $tableExists = $stmt->rowCount() > 0;

    if ($tableExists) {
        // Get the column information
        $columnsQuery = "SHOW COLUMNS FROM reports";
        $stmt = $pdo->prepare($columnsQuery);
        $stmt->execute();
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);

        // Log the columns for debugging
        error_log('Existing reports table columns: ' . implode(', ', $columns));

        // Check if we need to add the start_date and end_date columns
        if (!in_array('start_date', $columns) || !in_array('end_date', $columns)) {
            // Add the missing columns
            if (!in_array('start_date', $columns)) {
                $pdo->exec("ALTER TABLE reports ADD COLUMN start_date DATE AFTER type");
                error_log('Added start_date column to reports table');
            }

            if (!in_array('end_date', $columns)) {
                $pdo->exec("ALTER TABLE reports ADD COLUMN end_date DATE AFTER start_date");
                error_log('Added end_date column to reports table');
            }
        }
    } else {
        // Create reports table
        $createTableSql = "CREATE TABLE reports (
            id INT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255) NOT NULL,
            type VARCHAR(50) NOT NULL,
            start_date DATE NOT NULL,
            end_date DATE NOT NULL,
            filters TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            generated_by INT,
            data LONGTEXT
        )";

        $pdo->exec($createTableSql);
        error_log('Created new reports table');

        // Insert sample data
        $sampleData = [
            [
                'title' => 'Monthly Inspection Summary',
                'type' => 'inspection_summary',
                'start_date' => date('Y-m-d', strtotime('-30 days')),
                'end_date' => date('Y-m-d'),
                'filters' => json_encode(['vehicleIds' => [], 'userIds' => []]),
                'data' => json_encode([
                    'totalInspections' => 48,
                    'passRate' => 87,
                    'avgDuration' => 22,
                    'issuesFound' => 14,
                    'recentInspections' => [
                        ['id' => 101, 'vehicleName' => 'Tractor A-123', 'inspector' => 'John Mechanic', 'date' => date('Y-m-d', strtotime('-2 days')), 'status' => 'Pass'],
                        ['id' => 102, 'vehicleName' => 'Harvester B-456', 'inspector' => 'Sarah Technician', 'date' => date('Y-m-d', strtotime('-3 days')), 'status' => 'Fail']
                    ]
                ])
            ],
            [
                'title' => 'Q1 Defect Analysis',
                'type' => 'defect_analysis',
                'start_date' => date('Y-m-d', strtotime('-90 days')),
                'end_date' => date('Y-m-d'),
                'filters' => json_encode(['vehicleIds' => [], 'severityLevels' => ['Low', 'Medium', 'High']]),
                'data' => json_encode([
                    'totalDefects' => 37,
                    'openDefects' => 8,
                    'highSeverityCount' => 5,
                    'avgResolutionDays' => 3.2,
                    'commonDefects' => [
                        ['type' => 'Hydraulic System', 'count' => 12, 'avgSeverity' => 'Medium', 'percentage' => 32],
                        ['type' => 'Electrical', 'count' => 9, 'avgSeverity' => 'Low', 'percentage' => 24]
                    ]
                ])
            ]
        ];

        $insertSql = "INSERT INTO reports (title, type, start_date, end_date, filters, data) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $pdo->prepare($insertSql);

        foreach ($sampleData as $report) {
            $stmt->execute([
                $report['title'],
                $report['type'],
                $report['start_date'],
                $report['end_date'],
                $report['filters'],
                $report['data']
            ]);
        }
    }

    // GET request - Fetch reports
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // Get the column information to build a dynamic query
        $columnsQuery = "SHOW COLUMNS FROM reports";
        $stmt = $pdo->prepare($columnsQuery);
        $stmt->execute();
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);

        // Build a dynamic SELECT query based on available columns
        $selectColumns = ['id', 'title', 'type', 'created_at', 'data'];

        // Add optional columns if they exist
        if (in_array('start_date', $columns)) $selectColumns[] = 'start_date';
        if (in_array('end_date', $columns)) $selectColumns[] = 'end_date';
        if (in_array('filters', $columns)) $selectColumns[] = 'filters';
        if (in_array('generated_by', $columns)) $selectColumns[] = 'generated_by';

        $sql = "SELECT " . implode(', ', $selectColumns) . " FROM reports ORDER BY created_at DESC";
        $stmt = $pdo->prepare($sql);
        $stmt->execute();

        $reports = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Create a base report object
            $report = [
                'id' => $row['id'],
                'title' => $row['title'],
                'type' => $row['type'],
                'created_at' => $row['created_at'],
                'data' => json_decode($row['data'], true)
            ];

            // Add optional fields if they exist
            if (isset($row['start_date'])) {
                $report['startDate'] = $row['start_date'];
            } else {
                $report['startDate'] = date('Y-m-d', strtotime('-30 days')); // Default to 30 days ago
            }

            if (isset($row['end_date'])) {
                $report['endDate'] = $row['end_date'];
            } else {
                $report['endDate'] = date('Y-m-d'); // Default to today
            }

            if (isset($row['filters'])) {
                $report['filters'] = json_decode($row['filters'], true);
            } else {
                $report['filters'] = [];
            }

            $reports[] = $report;
        }

        echo json_encode($reports);
    }
    // POST request - Create a new report
    else if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Get the request body
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$data) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid request data']);
            exit();
        }

        // Validate required fields
        if (empty($data['title']) || empty($data['type']) || empty($data['startDate']) || empty($data['endDate'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields']);
            exit();
        }

        // Prepare filters and report data
        $filters = isset($data['filters']) ? json_encode($data['filters']) : '{}';

        // Generate report data based on type
        $reportData = generateReportData($data['type'], $data['startDate'], $data['endDate'], $data['filters'] ?? [], $pdo);

        // Get the column information to build a dynamic query
        $columnsQuery = "SHOW COLUMNS FROM reports";
        $stmt = $pdo->prepare($columnsQuery);
        $stmt->execute();
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);

        // Build the INSERT query dynamically based on available columns
        $insertColumns = ['title', 'type', 'data'];
        $insertValues = [$data['title'], $data['type'], json_encode($reportData)];
        $placeholders = ['?', '?', '?'];

        // Add optional columns if they exist
        if (in_array('start_date', $columns)) {
            $insertColumns[] = 'start_date';
            $insertValues[] = $data['startDate'];
            $placeholders[] = '?';
        }

        if (in_array('end_date', $columns)) {
            $insertColumns[] = 'end_date';
            $insertValues[] = $data['endDate'];
            $placeholders[] = '?';
        }

        if (in_array('filters', $columns)) {
            $insertColumns[] = 'filters';
            $insertValues[] = $filters;
            $placeholders[] = '?';
        }

        $sql = "INSERT INTO reports (" . implode(', ', $insertColumns) . ") VALUES (" . implode(', ', $placeholders) . ")";
        $stmt = $pdo->prepare($sql);

        $result = $stmt->execute($insertValues);

        if ($result) {
            $reportId = $pdo->lastInsertId();

            // Get the column information for the SELECT query
            $selectColumns = ['id', 'title', 'type', 'created_at', 'data'];

            // Add optional columns if they exist
            if (in_array('start_date', $columns)) $selectColumns[] = 'start_date';
            if (in_array('end_date', $columns)) $selectColumns[] = 'end_date';
            if (in_array('filters', $columns)) $selectColumns[] = 'filters';

            // Fetch the newly created report
            $sql = "SELECT " . implode(', ', $selectColumns) . " FROM reports WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$reportId]);

            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            // Create a base report object
            $newReport = [
                'id' => $row['id'],
                'title' => $row['title'],
                'type' => $row['type'],
                'created_at' => $row['created_at'],
                'data' => json_decode($row['data'], true)
            ];

            // Add optional fields if they exist
            if (isset($row['start_date'])) {
                $newReport['startDate'] = $row['start_date'];
            } else {
                $newReport['startDate'] = $data['startDate']; // Use the input data
            }

            if (isset($row['end_date'])) {
                $newReport['endDate'] = $row['end_date'];
            } else {
                $newReport['endDate'] = $data['endDate']; // Use the input data
            }

            if (isset($row['filters'])) {
                $newReport['filters'] = json_decode($row['filters'], true);
            } else {
                $newReport['filters'] = $data['filters'] ?? [];
            }

            echo json_encode(['success' => true, 'report' => $newReport]);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to create report']);
        }
    }
    // DELETE request - Delete a report
    else if ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
        // Get the request body
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$data || !isset($data['id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid request data']);
            exit();
        }

        // Check if the report exists first
        $checkSql = "SELECT id FROM reports WHERE id = ?";
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([$data['id']]);

        if ($checkStmt->rowCount() === 0) {
            // Report doesn't exist, but we'll return success anyway
            // This prevents issues if the client tries to delete a report that was already deleted
            echo json_encode(['success' => true, 'message' => 'Report not found, but operation considered successful']);
            exit();
        }

        // Delete the report
        $sql = "DELETE FROM reports WHERE id = ?";
        $stmt = $pdo->prepare($sql);

        $result = $stmt->execute([$data['id']]);

        if ($result) {
            echo json_encode(['success' => true]);
        } else {
            // Log the error for debugging
            error_log('Failed to delete report ID: ' . $data['id'] . ' - Error: ' . implode(', ', $stmt->errorInfo()));
            http_response_code(500);
            echo json_encode(['error' => 'Failed to delete report']);
        }
    }
    else {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    // Log the error server-side
    error_log('Error in reports.php: ' . $e->getMessage());

    // Return a clean error message to the client
    http_response_code(500);
    echo json_encode(['error' => 'Server error occurred: ' . $e->getMessage()]);
}

// Helper function to generate report data based on type
function generateReportData($type, $startDate, $endDate, $filters, $pdo) {
    switch ($type) {
        case 'inspection_summary':
            return generateInspectionSummary($startDate, $endDate, $filters, $pdo);
        case 'defect_analysis':
            return generateDefectAnalysis($startDate, $endDate, $filters, $pdo);
        case 'vehicle_history':
            return generateVehicleHistory($startDate, $endDate, $filters, $pdo);
        case 'maintenance_log':
            return generateMaintenanceLog($startDate, $endDate, $filters, $pdo);
        case 'user_activity':
            return generateUserActivity($startDate, $endDate, $filters, $pdo);
        case 'compliance_report':
            return generateComplianceReport($startDate, $endDate, $filters, $pdo);
        default:
            return [];
    }
}

// Helper functions to generate specific report types
function generateInspectionSummary($startDate, $endDate, $filters, $pdo) {
    // In a real app, this would query the database for actual inspection data
    // For now, we'll generate dummy data

    // Try to get some real data if available
    try {
        $sql = "SELECT COUNT(*) as count FROM inspections WHERE inspection_date BETWEEN ? AND ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$startDate, $endDate]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $totalInspections = $row['count'];
    } catch (Exception $e) {
        $totalInspections = rand(30, 80);
    }

    return [
        'totalInspections' => $totalInspections ?: rand(30, 80),
        'passRate' => rand(70, 100),
        'avgDuration' => rand(15, 30),
        'issuesFound' => rand(5, 25),
        'recentInspections' => [
            ['id' => rand(100, 999), 'vehicleName' => 'Tractor A-123', 'inspector' => 'John Mechanic', 'date' => date('Y-m-d', strtotime('-2 days')), 'status' => 'Pass'],
            ['id' => rand(100, 999), 'vehicleName' => 'Harvester B-456', 'inspector' => 'Sarah Technician', 'date' => date('Y-m-d', strtotime('-3 days')), 'status' => 'Fail'],
            ['id' => rand(100, 999), 'vehicleName' => 'Utility Vehicle C-789', 'inspector' => 'Mike Driver', 'date' => date('Y-m-d', strtotime('-4 days')), 'status' => 'Pass']
        ]
    ];
}

function generateDefectAnalysis($startDate, $endDate, $filters, $pdo) {
    // Try to get some real data if available
    try {
        $sql = "SELECT COUNT(*) as count FROM defects WHERE created_at BETWEEN ? AND ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$startDate, $endDate]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $totalDefects = $row['count'];
    } catch (Exception $e) {
        $totalDefects = rand(20, 60);
    }

    return [
        'totalDefects' => $totalDefects ?: rand(20, 60),
        'openDefects' => rand(5, 20),
        'highSeverityCount' => rand(1, 10),
        'avgResolutionDays' => round(rand(1, 7) + (rand(0, 9) / 10), 1),
        'commonDefects' => [
            ['type' => 'Hydraulic System', 'count' => rand(5, 15), 'avgSeverity' => 'Medium', 'percentage' => rand(20, 40)],
            ['type' => 'Electrical', 'count' => rand(5, 10), 'avgSeverity' => 'Low', 'percentage' => rand(15, 30)],
            ['type' => 'Engine', 'count' => rand(3, 8), 'avgSeverity' => 'High', 'percentage' => rand(10, 25)]
        ]
    ];
}

function generateVehicleHistory($startDate, $endDate, $filters, $pdo) {
    return [
        'vehicleId' => 'A-' . rand(100, 999),
        'makeModel' => 'John Deere 8R 410',
        'year' => rand(2018, 2023),
        'currentLocation' => 'North Yard',
        'status' => 'Active',
        'lastInspection' => date('Y-m-d', strtotime('-' . rand(1, 30) . ' days')),
        'maintenanceHistory' => [
            ['id' => rand(100, 999), 'type' => 'Oil Change', 'technician' => 'John Mechanic', 'date' => date('Y-m-d', strtotime('-15 days')), 'notes' => 'Routine maintenance'],
            ['id' => rand(100, 999), 'type' => 'Tire Rotation', 'technician' => 'Sarah Technician', 'date' => date('Y-m-d', strtotime('-45 days')), 'notes' => 'Replaced rear left tire']
        ]
    ];
}

function generateMaintenanceLog($startDate, $endDate, $filters, $pdo) {
    return [
        'totalServices' => rand(20, 50),
        'preventiveCount' => rand(10, 25),
        'repairCount' => rand(5, 20),
        'avgCost' => rand(200, 700),
        'maintenanceRecords' => [
            ['id' => rand(100, 999), 'vehicleName' => 'Tractor A-123', 'serviceType' => 'Oil Change', 'technician' => 'John Mechanic', 'date' => date('Y-m-d'), 'cost' => rand(100, 200), 'status' => 'Completed'],
            ['id' => rand(100, 999), 'vehicleName' => 'Harvester B-456', 'serviceType' => 'Engine Repair', 'technician' => 'Sarah Technician', 'date' => date('Y-m-d', strtotime('-5 days')), 'cost' => rand(500, 1000), 'status' => 'In Progress']
        ]
    ];
}

function generateUserActivity($startDate, $endDate, $filters, $pdo) {
    return [
        'activeUsers' => rand(5, 15),
        'inspectionCount' => rand(30, 80),
        'maintenanceCount' => rand(10, 40),
        'reportCount' => rand(5, 25),
        'userActivities' => [
            ['id' => rand(100, 999), 'userName' => 'John Mechanic', 'action' => 'Completed Inspection', 'date' => date('Y-m-d H:i:s'), 'details' => 'Tractor A-123 inspection completed'],
            ['id' => rand(100, 999), 'userName' => 'Sarah Technician', 'action' => 'Started Maintenance', 'date' => date('Y-m-d H:i:s', strtotime('-2 days')), 'details' => 'Harvester B-456 oil change']
        ]
    ];
}

function generateComplianceReport($startDate, $endDate, $filters, $pdo) {
    return [
        'complianceRate' => rand(80, 100),
        'violationCount' => rand(1, 10),
        'overdueCount' => rand(1, 5),
        'riskScore' => rand(1, 5),
        'complianceIssues' => [
            ['id' => rand(100, 999), 'title' => 'Overdue Inspection', 'vehicleName' => 'Tractor A-123', 'dateIdentified' => date('Y-m-d', strtotime('-10 days')), 'status' => 'Open', 'severity' => 'High'],
            ['id' => rand(100, 999), 'title' => 'Missing Maintenance Record', 'vehicleName' => 'Harvester B-456', 'dateIdentified' => date('Y-m-d', strtotime('-15 days')), 'status' => 'Resolved', 'severity' => 'Medium']
        ]
    ];
}

// Close database connection
$pdo = null;
?>