<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../../db_connect.php';

// Get data
$data = json_decode(file_get_contents("php://input"));

// Validate data
if (
    !isset($data->id) || !is_numeric($data->id) ||
    !isset($data->description) || empty($data->description) ||
    !isset($data->severity) || empty($data->severity) ||
    !isset($data->status) || empty($data->status)
) {
    http_response_code(400);
    echo json_encode(array("error" => "Incomplete or invalid defect data provided."));
    exit();
}

// Sanitize input
$id = intval($data->id);
$description = htmlspecialchars(strip_tags($data->description));
$severity = htmlspecialchars(strip_tags($data->severity));
$status = htmlspecialchars(strip_tags($data->status));
$assetId = isset($data->assetId) ? htmlspecialchars(strip_tags($data->assetId)) : null;
$inspectionId = isset($data->inspectionId) ? htmlspecialchars(strip_tags($data->inspectionId)) : null;

// Validate severity and status against allowed enum values (important!)
$allowed_severities = ['minor', 'major', 'critical'];
$allowed_statuses = ['open', 'in_progress', 'resolved'];

if (!in_array($severity, $allowed_severities)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid severity specified."));
    exit();
}

if (!in_array($status, $allowed_statuses)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid status specified."));
    exit();
}

try {
    // Check database connection
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception("Database connection object not available.");
    }

    // Prepare update statement
    $sql = "UPDATE defects SET description = :description, severity = :severity, status = :status, asset_id = :assetId, inspection_id = :inspectionId, updated_at = NOW() WHERE id = :id";
    $stmt = $pdo->prepare($sql);

    // Bind parameters
    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
    $stmt->bindParam(':description', $description);
    $stmt->bindParam(':severity', $severity);
    $stmt->bindParam(':status', $status);
    $stmt->bindParam(':assetId', $assetId);
    $stmt->bindParam(':inspectionId', $inspectionId);

    // Execute the query
    if ($stmt->execute()) {
        if ($stmt->rowCount() > 0) {
            http_response_code(200);
            echo json_encode(array("message" => "Defect updated successfully."));
        } else {
            http_response_code(404);
            echo json_encode(array("error" => "Defect not found."));
        }
    } else {
        http_response_code(503);
        echo json_encode(array("error" => "Unable to update defect. Database error."));
    }
} catch (PDOException $e) {
    http_response_code(503);
    echo json_encode(array("error" => "Database connection error: " . $e->getMessage()));
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(array("error" => "An unexpected error occurred: " . $e->getMessage()));
} finally {
    $pdo = null;
}
?>
