<?php
header("Access-Control-Allow-Origin: *"); // Allow requests from any origin (adjust for production)
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, OPTIONS"); // Allow POST and OPTIONS methods
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request (pre-flight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../../db_connect.php'; // Adjust path as needed

// Get posted data
$data = json_decode(file_get_contents("php://input"));

// Basic validation
if (
    !isset($data->id) ||
    !isset($data->name) || empty($data->name) ||
    !isset($data->email) || empty($data->email) || !filter_var($data->email, FILTER_VALIDATE_EMAIL) ||
    !isset($data->role) || empty($data->role) ||
    !isset($data->status) || empty($data->status)
) {
    http_response_code(400); // Bad Request
    echo json_encode(array("error" => "Incomplete or invalid user data provided."));
    exit();
}

// Sanitize input (basic example)
$id = filter_var($data->id, FILTER_SANITIZE_NUMBER_INT);
$name = htmlspecialchars(strip_tags($data->name));
$email = filter_var($data->email, FILTER_SANITIZE_EMAIL);
$role = htmlspecialchars(strip_tags($data->role));
$status = htmlspecialchars(strip_tags($data->status));

// Validate role and status against allowed enum values (important!)
$allowed_roles = ['owner', 'regional_manager', 'yard_manager', 'field_rep', 'dispatch', 'driver', 'wash_crew'];
$allowed_statuses = ['active', 'inactive', 'pending'];

if (!in_array($role, $allowed_roles)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid role specified."));
    exit();
}

if (!in_array($status, $allowed_statuses)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid status specified."));
    exit();
}


try {
    // $pdo is created in db_connect.php
    if (!isset($pdo) || !$pdo instanceof PDO) {
        // This condition might be hit if db_connect.php failed silently
        throw new Exception("Database connection object not available.");
    }

    // Check if email already exists for another user
    $checkEmailSql = "SELECT id FROM users WHERE email = :email AND id != :id";
    $checkEmailStmt = $pdo->prepare($checkEmailSql);
    $checkEmailStmt->bindParam(':email', $email);
    $checkEmailStmt->bindParam(':id', $id, PDO::PARAM_INT);
    $checkEmailStmt->execute();

    if ($checkEmailStmt->rowCount() > 0) {
        http_response_code(409); // Conflict
        echo json_encode(array("error" => "Email address is already in use by another user."));
        exit();
    }

    // Prepare update statement
    // Note: Password is not updated here. Handle password updates separately if needed.
    $sql = "UPDATE users SET name = :name, email = :email, role = :role, status = :status, updated_at = NOW() WHERE id = :id";
    $stmt = $pdo->prepare($sql);

    // Bind parameters
    $stmt->bindParam(':name', $name);
    $stmt->bindParam(':email', $email);
    $stmt->bindParam(':role', $role);
    $stmt->bindParam(':status', $status);
    $stmt->bindParam(':id', $id, PDO::PARAM_INT);

    // Execute the query
    if ($stmt->execute()) {
        if ($stmt->rowCount() > 0) {
            http_response_code(200); // OK
            echo json_encode(array("message" => "User updated successfully."));
        } else {
            // Technically OK, but no rows were affected (maybe ID didn't exist or data was the same)
            http_response_code(200); // Or potentially 404 if ID not found is critical
            echo json_encode(array("message" => "User found, but no changes were made or user not found."));
        }
    } else {
        http_response_code(503); // Service Unavailable (database error)
        echo json_encode(array("error" => "Unable to update user. Database error."));
    }

} catch (PDOException $e) {
    http_response_code(503); // Service Unavailable
    echo json_encode(array("error" => "Database connection error: " . $e->getMessage()));
} catch (Exception $e) {
    http_response_code(500); // Internal Server Error
    echo json_encode(array("error" => "An unexpected error occurred: " . $e->getMessage()));
} finally {
    $pdo = null; // Close connection
}
?>
