<?php
require_once '../db_connect.php'; // Adjust path as needed
header("Access-Control-Allow-Origin: *"); // Allow all origins (adjust for production)
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

$method = $_SERVER['REQUEST_METHOD'];

// Assuming region_id might be needed later, e.g., from session or request
// $region_id = $_SESSION['region_id'] ?? null; // Example: Get region_id from session
$region_id = 1; // Hardcoding Region ID 1 ('Central Region') for now, adjust as needed

if ($method == 'GET') {
    if (!$region_id) {
        http_response_code(403); // Forbidden or Bad Request
        echo json_encode(['error' => 'Region ID not specified.']);
        exit();
    }

    $summary = [];
    $breakdown = [];
    $issues = [];
    $error = null;

    try {
        // --- Fetch Compliance Summary ---
        // Overall Rate (Example: % approved inspections in last 30 days)
        $stmt_rate = $conn->prepare("
            SELECT 
                COUNT(i.id) AS inspectionsTotal,
                SUM(CASE WHEN i.status = 'approved' THEN 1 ELSE 0 END) AS inspectionsPassed
            FROM inspections i 
            JOIN yards y ON i.yard_id = y.id 
            WHERE y.region_id = ? AND i.inspection_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ");
        if (!$stmt_rate) throw new Exception("Prepare failed (rate): " . $conn->error);
        $stmt_rate->bind_param("i", $region_id);
        if (!$stmt_rate->execute()) throw new Exception("Execute failed (rate): " . $stmt_rate->error);
        $result_rate = $stmt_rate->get_result();
        $rate_data = $result_rate->fetch_assoc();
        $stmt_rate->close();

        $overallRate = '0%';
        $inspectionsTotal = (int)($rate_data['inspectionsTotal'] ?? 0);
        if ($inspectionsTotal > 0) {
            $overallRate = round(((int)($rate_data['inspectionsPassed'] ?? 0) * 100.0) / $inspectionsTotal, 0) . '%';
        }

        // Overdue Actions (Example: High/Critical maintenance tasks past due date)
        $stmt_overdue = $conn->prepare("
            SELECT COUNT(*) AS overdueCount 
            FROM maintenance_tasks mt
            JOIN vehicles v ON mt.vehicle_id = v.id
            JOIN yards y ON v.yard_id = y.id
            WHERE y.region_id = ? 
              AND mt.priority IN ('High', 'Critical') 
              AND mt.status IN ('Scheduled', 'Pending') -- Assuming Pending means not started
              AND mt.scheduled_date < CURDATE() 
        ");
         if (!$stmt_overdue) throw new Exception("Prepare failed (overdue): " . $conn->error);
         $stmt_overdue->bind_param("i", $region_id);
         if (!$stmt_overdue->execute()) throw new Exception("Execute failed (overdue): " . $stmt_overdue->error);
         $result_overdue = $stmt_overdue->get_result();
         $overdue_data = $result_overdue->fetch_assoc();
         $stmt_overdue->close();
         $overdueActions = (int)($overdue_data['overdueCount'] ?? 0);

        $summary = [
            'overallRate' => $overallRate,
            'inspectionsTotal' => $inspectionsTotal,
            'overdueActions' => $overdueActions
        ];

        // --- Fetch Compliance Breakdown ---
        // This requires more specific logic/data points not fully defined in the schema yet.
        // Using placeholders based on the dummy data structure.
        // Real implementation would involve queries for each area.
        $breakdown = [
            ['area' => 'Pre-Trip Inspections', 'rate' => '98%', 'status' => 'Good'], // Placeholder - Needs query
            ['area' => 'Maintenance Schedules', 'rate' => '92%', 'status' => 'Needs Attention'], // Placeholder - Needs query (e.g., % completed on time)
            ['area' => 'Driver Hours Logged', 'rate' => '99%', 'status' => 'Excellent'], // Placeholder - Needs external data or HOS table
            ['area' => 'Vehicle Registrations', 'rate' => '100%', 'status' => 'Excellent'], // Placeholder - Needs registration expiry data in vehicles table
        ];


        // --- Fetch Recent Compliance Issues ---
        // Example: Fetch recent overdue high-priority tasks
         $stmt_issues = $conn->prepare("
            SELECT 
                mt.id, -- Use maintenance task ID as issue ID for this example
                y.name AS yard_name, 
                CONCAT('Overdue Maintenance (', mt.priority, ')') AS issue_type, 
                v.vehicle_identifier, 
                NULL AS driver_name, -- Driver not directly linked to maintenance task in this schema
                mt.scheduled_date AS issue_date
            FROM maintenance_tasks mt
            JOIN vehicles v ON mt.vehicle_id = v.id
            JOIN yards y ON v.yard_id = y.id
            WHERE y.region_id = ? 
              AND mt.priority IN ('High', 'Critical') 
              AND mt.status IN ('Scheduled', 'Pending') 
              AND mt.scheduled_date < CURDATE() 
            ORDER BY mt.scheduled_date DESC
            LIMIT 5 -- Limit results
        ");
         if (!$stmt_issues) throw new Exception("Prepare failed (issues): " . $conn->error);
         $stmt_issues->bind_param("i", $region_id);
         if (!$stmt_issues->execute()) throw new Exception("Execute failed (issues): " . $stmt_issues->error);
         $result_issues = $stmt_issues->get_result();
         while ($row = $result_issues->fetch_assoc()) {
             // Format date
             if ($row['issue_date']) {
                 $date = new DateTime($row['issue_date']);
                 $row['issue_date'] = $date->format('Y-m-d');
             }
             $issues[] = $row;
         }
         $stmt_issues->close();
         
         // Add logic here to fetch other issue types like unaddressed flagged inspections if needed

    } catch (Exception $e) {
        $error = "Database error: " . $e->getMessage();
        http_response_code(500);
    }

    // --- Combine and Output ---
    if ($error) {
        echo json_encode(['error' => $error]);
    } else {
        echo json_encode(['summary' => $summary, 'breakdown' => $breakdown, 'issues' => $issues]);
    }

} else {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['error' => 'Method not allowed.']);
}

$conn->close();
?>