<?php
require_once '../db_connect.php'; // Adjust path as needed
header("Access-Control-Allow-Origin: *"); // Allow all origins (adjust for production)
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

$method = $_SERVER['REQUEST_METHOD'];

// Assuming region_id might be needed later, e.g., from session or request
// $region_id = $_SESSION['region_id'] ?? null; // Example: Get region_id from session
$region_id = 1; // Hardcoding Region ID 1 ('Central Region') for now, adjust as needed

if ($method == 'GET') {
    if (!$region_id) {
        http_response_code(403); // Forbidden or Bad Request
        echo json_encode(['error' => 'Region ID not specified.']);
        exit();
    }

    $summary = [];
    $by_yard = [];
    $attention = [];
    $error = null;

    try {
        // --- Fetch Fleet Summary ---
        $stmt_summary = $conn->prepare("
            SELECT 
                COUNT(*) AS totalVehicles,
                SUM(CASE WHEN v.status = 'Operational' OR v.status = 'Ready' OR v.status = 'In Use' OR v.status = 'On Route' THEN 1 ELSE 0 END) AS operational,
                SUM(CASE WHEN v.status = 'Maintenance' THEN 1 ELSE 0 END) AS inMaintenance,
                SUM(CASE WHEN v.status = 'Out of Service' THEN 1 ELSE 0 END) AS outOfService
            FROM vehicles v
            JOIN yards y ON v.yard_id = y.id
            WHERE y.region_id = ?
        ");
        if ($stmt_summary === false) throw new Exception("Prepare failed (summary): " . $conn->error);
        $stmt_summary->bind_param("i", $region_id);
        if (!$stmt_summary->execute()) throw new Exception("Execute failed (summary): " . $stmt_summary->error);
        $result_summary = $stmt_summary->get_result();
        $summary = $result_summary->fetch_assoc() ?: ['totalVehicles' => 0, 'operational' => 0, 'inMaintenance' => 0, 'outOfService' => 0];
        // Ensure counts are integers
        foreach ($summary as $key => $value) {
            $summary[$key] = (int)$value;
        }
        $stmt_summary->close();

        // --- Fetch Status by Yard ---
        $stmt_by_yard = $conn->prepare("
            SELECT 
                y.name AS yard_name,
                SUM(CASE WHEN v.status = 'Operational' OR v.status = 'Ready' OR v.status = 'In Use' OR v.status = 'On Route' THEN 1 ELSE 0 END) AS operational_count,
                SUM(CASE WHEN v.status = 'Maintenance' THEN 1 ELSE 0 END) AS maintenance_count,
                SUM(CASE WHEN v.status = 'Out of Service' THEN 1 ELSE 0 END) AS out_of_service_count
            FROM vehicles v
            JOIN yards y ON v.yard_id = y.id
            WHERE y.region_id = ?
            GROUP BY y.id, y.name
            ORDER BY y.name
        ");
        if ($stmt_by_yard === false) throw new Exception("Prepare failed (by yard): " . $conn->error);
        $stmt_by_yard->bind_param("i", $region_id);
        if (!$stmt_by_yard->execute()) throw new Exception("Execute failed (by yard): " . $stmt_by_yard->error);
        $result_by_yard = $stmt_by_yard->get_result();
        while ($row = $result_by_yard->fetch_assoc()) {
             // Ensure counts are integers
            $row['operational_count'] = (int)$row['operational_count'];
            $row['maintenance_count'] = (int)$row['maintenance_count'];
            $row['out_of_service_count'] = (int)$row['out_of_service_count'];
            $by_yard[] = $row;
        }
        $stmt_by_yard->close();

        // --- Fetch Vehicles Needing Attention ---
        // Example: Fetch vehicles with 'High' or 'Critical' priority maintenance tasks that are 'Scheduled' or 'In Progress'
        // Or vehicles with recent 'flagged' inspections not yet addressed. This logic needs refinement based on exact rules.
        $stmt_attention = $conn->prepare("
            SELECT 
                v.vehicle_identifier, 
                y.name AS yard_name, 
                mt.description AS issue_description, 
                mt.priority AS issue_priority 
            FROM maintenance_tasks mt
            JOIN vehicles v ON mt.vehicle_id = v.id
            JOIN yards y ON v.yard_id = y.id
            WHERE y.region_id = ? 
              AND mt.priority IN ('High', 'Critical') 
              AND mt.status IN ('Scheduled', 'In Progress')
            ORDER BY FIELD(mt.priority, 'Critical', 'High'), mt.scheduled_date ASC
            LIMIT 10 -- Limit results for performance
        ");
         if ($stmt_attention === false) throw new Exception("Prepare failed (attention): " . $conn->error);
         $stmt_attention->bind_param("i", $region_id);
         if (!$stmt_attention->execute()) throw new Exception("Execute failed (attention): " . $stmt_attention->error);
         $result_attention = $stmt_attention->get_result();
         while ($row = $result_attention->fetch_assoc()) {
             $attention[] = $row;
         }
         $stmt_attention->close();
         
         // Add logic here to fetch vehicles with recent unaddressed flagged inspections if needed

    } catch (Exception $e) {
        $error = $e->getMessage();
        http_response_code(500);
    }

    // --- Combine and Output ---
    if ($error) {
        echo json_encode(['error' => $error]);
    } else {
        echo json_encode(['summary' => $summary, 'byYard' => $by_yard, 'attention' => $attention]);
    }

} else {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['error' => 'Method not allowed.']);
}

$conn->close();
?>