<?php
require_once '../db_connect.php'; // Adjust path as needed
header("Access-Control-Allow-Origin: *"); // Allow all origins (adjust for production)
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

$method = $_SERVER['REQUEST_METHOD'];

// Assuming region_id might be needed later, e.g., from session or request
// $region_id = $_SESSION['region_id'] ?? null; // Example: Get region_id from session
$region_id = 1; // Hardcoding Region ID 1 ('Central Region') for now, adjust as needed

if ($method == 'GET') {
    if (!$region_id) {
        http_response_code(403); // Forbidden or Bad Request
        echo json_encode(['error' => 'Region ID not specified.']);
        exit();
    }

    $overview = [];
    $yards_summary = [];
    $error = null;

    try {
        // --- Fetch Overview Stats ---
        $stmt_overview = $conn->prepare("
            SELECT 
                r.name AS regionName,
                (SELECT COUNT(*) FROM yards WHERE region_id = r.id) AS totalYards,
                (SELECT COUNT(*) FROM vehicles v JOIN yards y ON v.yard_id = y.id WHERE y.region_id = r.id) AS totalVehicles,
                (SELECT COUNT(DISTINCT va.user_id) 
                 FROM vehicle_assignments va 
                 JOIN users u ON va.user_id = u.id 
                 JOIN vehicles v ON va.vehicle_id = v.id
                 JOIN yards y ON v.yard_id = y.id
                 WHERE u.role = 'driver' AND va.status = 'active' AND y.region_id = r.id) AS activeDrivers,
                -- Compliance Rate (Example: % of inspections approved in the last 30 days)
                (SELECT ROUND(
                    (SUM(CASE WHEN i.status = 'approved' THEN 1 ELSE 0 END) * 100.0) / COUNT(i.id), 0) 
                 FROM inspections i 
                 JOIN yards y ON i.yard_id = y.id 
                 WHERE y.region_id = r.id AND i.inspection_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
                ) AS complianceRatePercentage
            FROM regions r 
            WHERE r.id = ?
        ");
        
        if ($stmt_overview === false) throw new Exception("Prepare failed (overview): " . $conn->error);
        $stmt_overview->bind_param("i", $region_id);
        if (!$stmt_overview->execute()) throw new Exception("Execute failed (overview): " . $stmt_overview->error);
        
        $result_overview = $stmt_overview->get_result();
        if ($result_overview->num_rows > 0) {
            $overview = $result_overview->fetch_assoc();
            // Format compliance rate
            $overview['complianceRate'] = ($overview['complianceRatePercentage'] ?? 0) . '%';
            unset($overview['complianceRatePercentage']); // Remove the raw percentage
        } else {
             $overview = ['regionName' => 'Unknown Region', 'totalYards' => 0, 'totalVehicles' => 0, 'activeDrivers' => 0, 'complianceRate' => 'N/A'];
        }
        $stmt_overview->close();

        // --- Fetch Yard Summaries ---
        $stmt_yards = $conn->prepare("
            SELECT 
                y.id, 
                y.name, 
                -- Determine yard status based on some logic (e.g., presence of high-priority issues) - Placeholder
                'Operational' AS status, -- Placeholder status
                (SELECT COUNT(*) FROM vehicles WHERE yard_id = y.id) AS vehicle_count,
                (SELECT COUNT(*) FROM inspections WHERE yard_id = y.id AND status = 'pending') AS pending_inspections
            FROM yards y
            WHERE y.region_id = ?
            ORDER BY y.name
        ");

        if ($stmt_yards === false) throw new Exception("Prepare failed (yards): " . $conn->error);
        $stmt_yards->bind_param("i", $region_id);
        if (!$stmt_yards->execute()) throw new Exception("Execute failed (yards): " . $stmt_yards->error);

        $result_yards = $stmt_yards->get_result();
        while ($row = $result_yards->fetch_assoc()) {
            $yards_summary[] = $row;
        }
        $stmt_yards->close();

    } catch (Exception $e) {
        $error = $e->getMessage();
        http_response_code(500);
    }

    // --- Combine and Output ---
    if ($error) {
        echo json_encode(['error' => $error]);
    } else {
        echo json_encode(['overview' => $overview, 'yards' => $yards_summary]);
    }

} else {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['error' => 'Method not allowed.']);
}

$conn->close();
?>