<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once 'db_connect.php';

// Get the action parameter
$action = isset($_GET['action']) ? $_GET['action'] : '';

// Set content type to JSON
header('Content-Type: application/json');

// Handle different actions
switch ($action) {
    case 'getUserList':
        getUserList($pdo);
        break;
    case 'getDashboardSummary':
        getDashboardSummary($pdo);
        break;
    case 'getVehicleListSM':
        getVehicleListSM($pdo);
        break;
    case 'inviteTeamMember':
        inviteTeamMember($pdo);
        break;
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action specified']);
        break;
}

// Function to get list of users
function getUserList($pdo) {
    $created_by = isset($_GET['created_by']) ? $_GET['created_by'] : null;
    $role_filter = isset($_GET['role']) ? $_GET['role'] : null;

    error_log("getUserList called with created_by: " . ($created_by ? $created_by : 'null') . ", role: " . ($role_filter ? $role_filter : 'null'));

    try {
        // Build query based on parameters
        $query = "SELECT id, name, email, role, status, created_at, created_by FROM users WHERE 1=1";
        $params = [];

        // Filter by creator if specified
        if ($created_by) {
            $query .= " AND created_by = ?";
            $params[] = $created_by;
            error_log("Adding strict created_by filter: " . $created_by);
        }

        // Filter by role if specified
        if ($role_filter) {
            $query .= " AND role = ?";
            $params[] = $role_filter;
            error_log("Adding role filter: " . $role_filter);
        }

        $query .= " ORDER BY created_at DESC";
        error_log("Final SQL query: " . $query);

        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

        error_log("Found " . count($users) . " users matching criteria");
        echo json_encode(['success' => true, 'data' => $users]);
    } catch (PDOException $e) {
        error_log("Database error in getUserList: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

// Function to get dashboard summary
function getDashboardSummary($pdo) {
    $userId = isset($_GET['user_id']) ? $_GET['user_id'] : null;

    try {
        // Get total vehicles
        $vehicleStmt = $pdo->prepare("SELECT COUNT(*) as total FROM vehicles");
        $vehicleStmt->execute();
        $vehicleCount = $vehicleStmt->fetch(PDO::FETCH_ASSOC)['total'];

        // Get total users
        $userStmt = $pdo->prepare("SELECT COUNT(*) as total FROM users");
        $userStmt->execute();
        $userCount = $userStmt->fetch(PDO::FETCH_ASSOC)['total'];

        // Get total inspections
        $inspectionStmt = $pdo->prepare("SELECT COUNT(*) as total FROM inspections");
        $inspectionStmt->execute();
        $inspectionCount = $inspectionStmt->fetch(PDO::FETCH_ASSOC)['total'];

        // Get recent activities
        $activityStmt = $pdo->prepare("
            SELECT 'inspection' as type, i.id, v.name as vehicle_name, i.inspection_date as date, i.status
            FROM inspections i
            JOIN vehicles v ON i.vehicle_id = v.id
            ORDER BY i.inspection_date DESC
            LIMIT 5
        ");
        $activityStmt->execute();
        $activities = $activityStmt->fetchAll(PDO::FETCH_ASSOC);

        $summary = [
            'vehicleCount' => $vehicleCount,
            'userCount' => $userCount,
            'inspectionCount' => $inspectionCount,
            'recentActivities' => $activities
        ];

        echo json_encode(['success' => true, 'data' => $summary]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

// Function to get vehicle list (small cards)
function getVehicleListSM($pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT v.id, v.name, v.type, v.status, v.year, y.name as yard_name
            FROM vehicles v
            LEFT JOIN yards y ON v.yard_id = y.id
            ORDER BY v.id DESC
            LIMIT 10
        ");
        $stmt->execute();
        $vehicles = $stmt->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode(['success' => true, 'data' => $vehicles]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

// Function to invite team member
function inviteTeamMember($pdo) {
    // Get JSON data
    $data = json_decode(file_get_contents('php://input'), true);

    if (!isset($data['email']) || !isset($data['role'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Email and role are required']);
        return;
    }

    $email = $data['email'];
    $role = $data['role'];
    $name = isset($data['name']) ? $data['name'] : '';
    $created_by = isset($data['created_by']) ? $data['created_by'] : null;

    // Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid email format']);
        return;
    }

    try {
        // Check if email already exists
        $checkStmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $checkStmt->execute([$email]);

        if ($checkStmt->rowCount() > 0) {
            http_response_code(400);
            echo json_encode(['error' => 'Email already exists']);
            return;
        }

        // Generate a temporary password
        $tempPassword = bin2hex(random_bytes(4));
        $hashedPassword = password_hash($tempPassword, PASSWORD_BCRYPT);

        // Insert new user with created_by if provided
        if ($created_by) {
            $stmt = $pdo->prepare("INSERT INTO users (name, email, password, role, status, created_by) VALUES (?, ?, ?, ?, 'pending', ?)");
            $stmt->execute([$name, $email, $hashedPassword, $role, $created_by]);
        } else {
            $stmt = $pdo->prepare("INSERT INTO users (name, email, password, role, status) VALUES (?, ?, ?, ?, 'pending')");
            $stmt->execute([$name, $email, $hashedPassword, $role]);
        }

        if ($stmt->rowCount() > 0) {
            // In a real application, you would send an email with the temporary password
            echo json_encode([
                'success' => true,
                'message' => 'Invitation sent successfully',
                'tempPassword' => $tempPassword // Only for demo purposes
            ]);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to create user']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}
?>
