<?php
require_once '../db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get request body
$data = json_decode(file_get_contents('php://input'), true);

// Check if username is provided
if (!isset($data['username'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Username is required']);
    exit;
}

$username = $data['username'];

try {
    // Check if user exists and biometric is enabled
    $stmt = $pdo->prepare("SELECT * FROM users WHERE user_identifier = ? AND biometric_enabled = 1 LIMIT 1");
    $stmt->execute([$username]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // Log failed login attempt
        logActivity($pdo, null, 'biometric_login_failed', "Failed biometric login attempt for username: $username (user not found or biometric not enabled)");

        // Return error
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Biometric authentication failed']);
        exit;
    }

    // Check if account is locked
    if ($user['status'] === 'locked') {
        logActivity($pdo, $user['id'], 'biometric_login_failed', "Failed biometric login attempt for user ID: {$user['id']} (account locked)");

        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Account is locked. Please contact your administrator.']);
        exit;
    }

    // Check if account is inactive
    if ($user['status'] !== 'active') {
        logActivity($pdo, $user['id'], 'biometric_login_failed', "Failed biometric login attempt for user ID: {$user['id']} (account inactive)");

        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Account is not active. Please contact your administrator.']);
        exit;
    }

    // Generate token
    $token = generateToken($user['id']);

    // Update last login timestamp
    updateLastLogin($pdo, $user['id']);

    // Log successful login
    logActivity($pdo, $user['id'], 'biometric_login', "Successful biometric login for user ID: {$user['id']}");

    // Prepare user data for response (exclude sensitive fields)
    $userData = [
        'id' => $user['id'],
        'username' => $user['user_identifier'],
        'name' => $user['name'],
        'email' => $user['email'],
        'role' => $user['role'],
        'created_at' => $user['created_at'],
        'last_login' => isset($user['updated_at']) ? $user['updated_at'] : null,
        'biometric_enabled' => (bool)$user['biometric_enabled']
    ];

    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'Biometric login successful',
        'user' => $userData,
        'token' => $token
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}

// Helper functions

function generateToken($userId) {
    // In a real app, use a proper JWT library
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode([
        'user_id' => $userId,
        'iat' => time(),
        'exp' => time() + (60 * 60 * 24) // 24 hours
    ]);

    $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));

    $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, 'your_secret_key', true);
    $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));

    return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
}

function updateLastLogin($pdo, $userId) {
    $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->execute([$userId]);
}

function logActivity($pdo, $userId, $activityType, $details) {
    try {
        $stmt = $pdo->prepare("INSERT INTO activity_logs (user_id, activity_type, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$userId, $activityType, $details, $_SERVER['REMOTE_ADDR']]);
    } catch (PDOException $e) {
        // Silently fail - don't let logging errors affect the main functionality
        error_log("Error logging activity: " . $e->getMessage());
    }
}
?>
