<?php
require_once '../db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get request body
$data = json_decode(file_get_contents('php://input'), true);

// Check if email is provided
if (!isset($data['email'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Email is required']);
    exit;
}

$email = $data['email'];

// Validate email format
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid email format']);
    exit;
}

try {
    // Check if user exists with this email
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? LIMIT 1");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // For security reasons, don't reveal that the email doesn't exist
        // Instead, pretend we sent the email
        echo json_encode(['success' => true, 'message' => 'If your email is registered, you will receive PIN reset instructions shortly.']);
        exit;
    }

    // Generate a reset token
    $resetToken = bin2hex(random_bytes(32));
    $tokenExpiry = date('Y-m-d H:i:s', strtotime('+1 hour'));

    // Store the reset token in the database
    $stmt = $pdo->prepare("UPDATE users SET reset_token = ?, reset_token_expiry = ? WHERE id = ?");
    $stmt->execute([$resetToken, $tokenExpiry, $user['id']]);

    // Also update the password directly for testing purposes
    // In a real app, this would be done through a separate reset confirmation process
    $newPin = '1234'; // Default PIN for testing
    $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
    $stmt->execute([$newPin, $user['id']]);

    // Log the PIN reset request
    logActivity($pdo, $user['id'], 'pin_reset_request', "PIN reset requested for user ID: {$user['id']}");

    // In a real app, send an email with the reset link
    // For this demo, we'll just log it
    $resetLink = "https://yourapp.com/reset-pin?token=$resetToken";
    error_log("PIN Reset Link for {$user['email']}: $resetLink");

    // Return success response
    echo json_encode(['success' => true, 'message' => 'If your email is registered, you will receive PIN reset instructions shortly.']);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}

function logActivity($pdo, $userId, $activityType, $details) {
    try {
        $stmt = $pdo->prepare("INSERT INTO activity_logs (user_id, activity_type, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$userId, $activityType, $details, $_SERVER['REMOTE_ADDR']]);
    } catch (PDOException $e) {
        // Silently fail - don't let logging errors affect the main functionality
        error_log("Error logging activity: " . $e->getMessage());
    }
}
?>
