<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight CORS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../db_connect.php';

// Check if report ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Content-Type: application/json");
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Report ID is required']);
    exit();
}

$reportId = $_GET['id'];
$format = isset($_GET['format']) ? strtolower($_GET['format']) : 'pdf';

try {
    // Fetch report data from database
    $stmt = $conn->prepare("SELECT * FROM reports WHERE id = ?");
    $stmt->bind_param("i", $reportId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        header("Content-Type: application/json");
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Report not found']);
        exit();
    }
    
    $report = $result->fetch_assoc();
    $reportData = json_decode($report['data'], true);
    
    // Generate PDF using TCPDF
    if ($format === 'pdf') {
        // Include TCPDF library
        require_once('../tcpdf/tcpdf.php');
        
        // Create new PDF document
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        
        // Set document information
        $pdf->SetCreator('AgriScan');
        $pdf->SetAuthor('AgriScan System');
        $pdf->SetTitle($report['title']);
        $pdf->SetSubject('Vehicle Inspection Report');
        
        // Set default header data
        $pdf->SetHeaderData('', 0, 'AgriScan Report', $report['title']);
        
        // Set margins
        $pdf->SetMargins(15, 15, 15);
        $pdf->SetHeaderMargin(5);
        $pdf->SetFooterMargin(10);
        
        // Set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, 15);
        
        // Add a page
        $pdf->AddPage();
        
        // Set font
        $pdf->SetFont('helvetica', '', 12);
        
        // Report header
        $pdf->SetFont('helvetica', 'B', 16);
        $pdf->Cell(0, 10, $report['title'], 0, 1, 'C');
        $pdf->SetFont('helvetica', '', 12);
        $pdf->Ln(5);
        
        // Report metadata
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->Cell(40, 7, 'Report Type:', 0, 0);
        $pdf->SetFont('helvetica', '', 12);
        $pdf->Cell(0, 7, $report['type'], 0, 1);
        
        $pdf->SetFont('helvetica', 'B', 12);
        $pdf->Cell(40, 7, 'Generated On:', 0, 0);
        $pdf->SetFont('helvetica', '', 12);
        $pdf->Cell(0, 7, date('Y-m-d H:i:s', strtotime($report['created_at'])), 0, 1);
        
        if (isset($report['start_date']) && isset($report['end_date'])) {
            $pdf->SetFont('helvetica', 'B', 12);
            $pdf->Cell(40, 7, 'Date Range:', 0, 0);
            $pdf->SetFont('helvetica', '', 12);
            $pdf->Cell(0, 7, date('Y-m-d', strtotime($report['start_date'])) . ' to ' . date('Y-m-d', strtotime($report['end_date'])), 0, 1);
        }
        
        $pdf->Ln(10);
        
        // Report content based on type
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->Cell(0, 10, 'Report Summary', 0, 1);
        $pdf->SetFont('helvetica', '', 12);
        
        // Generate content based on report type
        switch ($report['type']) {
            case 'inspection_summary':
                generateInspectionSummaryContent($pdf, $reportData);
                break;
            case 'defect_analysis':
                generateDefectAnalysisContent($pdf, $reportData);
                break;
            case 'vehicle_history':
                generateVehicleHistoryContent($pdf, $reportData);
                break;
            case 'maintenance_log':
                generateMaintenanceLogContent($pdf, $reportData);
                break;
            case 'user_activity':
                generateUserActivityContent($pdf, $reportData);
                break;
            case 'compliance_report':
                generateComplianceReportContent($pdf, $reportData);
                break;
            default:
                $pdf->Cell(0, 10, 'No specific content for this report type.', 0, 1);
                break;
        }
        
        // Output the PDF
        $pdfFileName = 'report_' . $reportId . '.pdf';
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="' . $pdfFileName . '"');
        echo $pdf->Output($pdfFileName, 'S');
        exit();
    } 
    // Generate CSV
    else if ($format === 'csv') {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="report_' . $reportId . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // CSV headers based on report type
        switch ($report['type']) {
            case 'inspection_summary':
                fputcsv($output, ['Total Inspections', 'Pass Rate', 'Average Duration', 'Issues Found']);
                if (isset($reportData['totalInspections'])) {
                    fputcsv($output, [$reportData['totalInspections'], $reportData['passRate'] . '%', $reportData['avgDuration'] . ' min', $reportData['issuesFound']]);
                }
                break;
            // Add other report types as needed
            default:
                fputcsv($output, ['Report Type', 'Generated Date', 'Title']);
                fputcsv($output, [$report['type'], $report['created_at'], $report['title']]);
                break;
        }
        
        fclose($output);
        exit();
    } 
    // JSON format
    else {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'data' => [
                'id' => $report['id'],
                'title' => $report['title'],
                'type' => $report['type'],
                'created_at' => $report['created_at'],
                'data' => $reportData
            ]
        ]);
        exit();
    }
    
} catch (Exception $e) {
    header("Content-Type: application/json");
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error exporting report: ' . $e->getMessage()
    ]);
}

// Helper functions for generating report content
function generateInspectionSummaryContent($pdf, $data) {
    if (isset($data['totalInspections'])) {
        $pdf->Cell(0, 10, 'Total Inspections: ' . $data['totalInspections'], 0, 1);
        $pdf->Cell(0, 10, 'Pass Rate: ' . $data['passRate'] . '%', 0, 1);
        $pdf->Cell(0, 10, 'Average Duration: ' . $data['avgDuration'] . ' minutes', 0, 1);
        $pdf->Cell(0, 10, 'Issues Found: ' . $data['issuesFound'], 0, 1);
        
        if (isset($data['recentInspections']) && is_array($data['recentInspections'])) {
            $pdf->Ln(5);
            $pdf->SetFont('helvetica', 'B', 12);
            $pdf->Cell(0, 10, 'Recent Inspections', 0, 1);
            $pdf->SetFont('helvetica', '', 10);
            
            // Table header
            $pdf->SetFillColor(230, 230, 230);
            $pdf->Cell(20, 7, 'ID', 1, 0, 'C', 1);
            $pdf->Cell(60, 7, 'Vehicle', 1, 0, 'C', 1);
            $pdf->Cell(60, 7, 'Inspector', 1, 0, 'C', 1);
            $pdf->Cell(30, 7, 'Date', 1, 0, 'C', 1);
            $pdf->Cell(20, 7, 'Status', 1, 1, 'C', 1);
            
            // Table data
            foreach ($data['recentInspections'] as $inspection) {
                $pdf->Cell(20, 7, $inspection['id'], 1, 0, 'C');
                $pdf->Cell(60, 7, $inspection['vehicleName'], 1, 0, 'L');
                $pdf->Cell(60, 7, $inspection['inspector'], 1, 0, 'L');
                $pdf->Cell(30, 7, $inspection['date'], 1, 0, 'C');
                $pdf->Cell(20, 7, $inspection['status'], 1, 1, 'C');
            }
        }
    } else {
        $pdf->Cell(0, 10, 'No inspection summary data available.', 0, 1);
    }
}

function generateDefectAnalysisContent($pdf, $data) {
    if (isset($data['totalDefects'])) {
        $pdf->Cell(0, 10, 'Total Defects: ' . $data['totalDefects'], 0, 1);
        $pdf->Cell(0, 10, 'Open Defects: ' . $data['openDefects'], 0, 1);
        $pdf->Cell(0, 10, 'High Severity Defects: ' . $data['highSeverityCount'], 0, 1);
        $pdf->Cell(0, 10, 'Average Resolution Time: ' . $data['avgResolutionDays'] . ' days', 0, 1);
        
        if (isset($data['commonDefects']) && is_array($data['commonDefects'])) {
            $pdf->Ln(5);
            $pdf->SetFont('helvetica', 'B', 12);
            $pdf->Cell(0, 10, 'Common Defect Types', 0, 1);
            $pdf->SetFont('helvetica', '', 10);
            
            // Table header
            $pdf->SetFillColor(230, 230, 230);
            $pdf->Cell(60, 7, 'Defect Type', 1, 0, 'C', 1);
            $pdf->Cell(30, 7, 'Count', 1, 0, 'C', 1);
            $pdf->Cell(40, 7, 'Avg Severity', 1, 0, 'C', 1);
            $pdf->Cell(30, 7, 'Percentage', 1, 1, 'C', 1);
            
            // Table data
            foreach ($data['commonDefects'] as $defect) {
                $pdf->Cell(60, 7, $defect['type'], 1, 0, 'L');
                $pdf->Cell(30, 7, $defect['count'], 1, 0, 'C');
                $pdf->Cell(40, 7, $defect['avgSeverity'], 1, 0, 'C');
                $pdf->Cell(30, 7, $defect['percentage'] . '%', 1, 1, 'C');
            }
        }
    } else {
        $pdf->Cell(0, 10, 'No defect analysis data available.', 0, 1);
    }
}

function generateVehicleHistoryContent($pdf, $data) {
    // Implement vehicle history report content
    $pdf->Cell(0, 10, 'Vehicle history report content will be displayed here.', 0, 1);
}

function generateMaintenanceLogContent($pdf, $data) {
    // Implement maintenance log report content
    $pdf->Cell(0, 10, 'Maintenance log report content will be displayed here.', 0, 1);
}

function generateUserActivityContent($pdf, $data) {
    // Implement user activity report content
    $pdf->Cell(0, 10, 'User activity report content will be displayed here.', 0, 1);
}

function generateComplianceReportContent($pdf, $data) {
    if (isset($data['complianceRate'])) {
        $pdf->Cell(0, 10, 'Compliance Rate: ' . $data['complianceRate'] . '%', 0, 1);
        $pdf->Cell(0, 10, 'Violation Count: ' . $data['violationCount'], 0, 1);
        $pdf->Cell(0, 10, 'Overdue Items: ' . $data['overdueCount'], 0, 1);
        $pdf->Cell(0, 10, 'Risk Score: ' . $data['riskScore'] . '/5', 0, 1);
        
        if (isset($data['complianceIssues']) && is_array($data['complianceIssues'])) {
            $pdf->Ln(5);
            $pdf->SetFont('helvetica', 'B', 12);
            $pdf->Cell(0, 10, 'Compliance Issues', 0, 1);
            $pdf->SetFont('helvetica', '', 10);
            
            // Table header
            $pdf->SetFillColor(230, 230, 230);
            $pdf->Cell(20, 7, 'ID', 1, 0, 'C', 1);
            $pdf->Cell(60, 7, 'Issue', 1, 0, 'C', 1);
            $pdf->Cell(40, 7, 'Vehicle', 1, 0, 'C', 1);
            $pdf->Cell(30, 7, 'Date', 1, 0, 'C', 1);
            $pdf->Cell(20, 7, 'Status', 1, 0, 'C', 1);
            $pdf->Cell(20, 7, 'Severity', 1, 1, 'C', 1);
            
            // Table data
            foreach ($data['complianceIssues'] as $issue) {
                $pdf->Cell(20, 7, $issue['id'], 1, 0, 'C');
                $pdf->Cell(60, 7, $issue['title'], 1, 0, 'L');
                $pdf->Cell(40, 7, $issue['vehicleName'], 1, 0, 'L');
                $pdf->Cell(30, 7, $issue['dateIdentified'], 1, 0, 'C');
                $pdf->Cell(20, 7, $issue['status'], 1, 0, 'C');
                $pdf->Cell(20, 7, $issue['severity'], 1, 1, 'C');
            }
        }
    } else {
        $pdf->Cell(0, 10, 'No compliance report data available.', 0, 1);
    }
}
?>


