<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../db_connect.php';

// Get data
$data = json_decode(file_get_contents("php://input"));

// Validate data
if (
    !isset($data->assetId) || !is_numeric($data->assetId) ||
    !isset($data->templateId) || !is_numeric($data->templateId) ||
    !isset($data->inspectorId) || !is_numeric($data->inspectorId)
) {
    http_response_code(400);
    echo json_encode(array("success" => false, "message" => "Incomplete or invalid inspection data provided."));
    exit();
}

// Sanitize input
$assetId = intval($data->assetId);
$templateId = intval($data->templateId);
$inspectorId = intval($data->inspectorId);

try {
    // Check database connection
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception("Database connection object not available.");
    }

    // Fetch yard_id for the selected vehicle
    $yardSql = "SELECT yard_id FROM vehicles WHERE id = :assetId";
    $yardStmt = $pdo->prepare($yardSql);
    $yardStmt->bindParam(':assetId', $assetId, PDO::PARAM_INT);
    $yardStmt->execute();
    $vehicleData = $yardStmt->fetch(PDO::FETCH_ASSOC);
    $yardId = $vehicleData ? $vehicleData['yard_id'] : null; // Get yard_id or null if not found

    // Generate a unique inspection identifier (Example: INSP + timestamp)
    $inspectionIdentifier = 'INSP' . time(); 
    
    // Initialize findings (e.g., as an empty JSON object)
    $findings = '{}'; 

    // Prepare insert statement including the new fields
    $sql = "INSERT INTO inspections (inspection_identifier, vehicle_id, yard_id, template_id, inspector_id, findings, inspection_date, status) 
            VALUES (:inspectionIdentifier, :assetId, :yardId, :templateId, :inspectorId, :findings, CURDATE(), 'pending')";
    $stmt = $pdo->prepare($sql);

    // Bind parameters
    $stmt->bindParam(':inspectionIdentifier', $inspectionIdentifier);
    $stmt->bindParam(':assetId', $assetId, PDO::PARAM_INT);
    $stmt->bindParam(':yardId', $yardId, PDO::PARAM_INT); // Bind yard_id
    $stmt->bindParam(':templateId', $templateId, PDO::PARAM_INT);
    $stmt->bindParam(':inspectorId', $inspectorId, PDO::PARAM_INT);
    $stmt->bindParam(':findings', $findings); // Bind findings

    // Execute the query
    if ($stmt->execute()) {
        http_response_code(200);
        echo json_encode(array("success" => true, "message" => "Inspection started successfully."));
    } else {
        http_response_code(503);
        echo json_encode(array("success" => false, "message" => "Unable to start inspection. Database error."));
    }
} catch (PDOException $e) {
    http_response_code(503);
    echo json_encode(array("success" => false, "message" => "Database connection error: " . $e->getMessage()));
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(array("success" => false, "message" => "An unexpected error occurred: " . $e->getMessage()));
} finally {
    $pdo = null;
}
?>
