<?php
// Set CORS headers to allow frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database connection
require_once 'db_connect.php';

// Get the action parameter from the request
$action = $_GET['action'] ?? '';

// Set JSON content type
header('Content-Type: application/json');

// Route the request based on action
switch ($action) {
    case 'getDashboard':
        getDashboard($conn);
        break;
    case 'getTasks':
        getTasks($conn);
        break;
    case 'getAlerts':
        getAlerts($conn);
        break;
    case 'getFleetStatus':
        getFleetStatus($conn);
        break;
    case 'getInspectionStats':
        getInspectionStats($conn);
        break;
    case 'getUpcomingMaintenance':
        getUpcomingMaintenance($conn);
        break;
    case 'getUserProfile':
        getUserProfile($conn);
        break;
    case 'getRecentInspections':
        getRecentInspections($conn);
        break;
    case 'getFleetLocations':
        getFleetLocations($conn);
        break;
    default:
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid action']);
        break;
}

/**
 * Get all dashboard data in a single API call
 */
function getDashboard($conn) {
    try {
        // Get all dashboard data in a single API call
        $response = [
            'fleetStatus' => getFleetStatusData($conn),
            'tasks' => getTasksData($conn),
            'alerts' => getAlertsData($conn),
            'inspectionStats' => getInspectionStatsData($conn),
            'upcomingMaintenance' => getUpcomingMaintenanceData($conn),
            'userProfile' => getUserProfileData($conn, 7), // Hardcoded for Dispatcher One (ID 7)
            'recentInspections' => getRecentInspectionsData($conn),
            'quickStats' => getQuickStatsData($conn)
        ];

        echo json_encode($response);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching dashboard data: ' . $e->getMessage()]);
    }
}

/**
 * Fetch pending tasks (inspections with status 'pending')
 */
function getTasks($conn) {
    try {
        $tasks = getTasksData($conn);
        echo json_encode($tasks);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching tasks: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get tasks data
 */
function getTasksData($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT i.id, v.name AS vehicle, u.name AS assignee
            FROM inspections i
            JOIN vehicles v ON i.vehicle_id = v.id
            JOIN users u ON i.inspector_id = u.id
            WHERE i.status = 'pending'
        ");
        $stmt->execute();
        $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Format tasks with title and priority
        foreach ($tasks as &$task) {
            $task['title'] = "Inspect " . $task['vehicle'];
            $task['priority'] = 'Medium'; // Hardcoded, as priority isn't in schema
            $task['status'] = 'pending'; // Add status field
        }

        return $tasks;
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            ['id' => 1, 'title' => 'Assign driver to Route #42', 'description' => 'Need to assign a driver to the morning delivery route', 'status' => 'pending', 'priority' => 'High', 'due_date' => '2025-03-20', 'assignee' => null],
            ['id' => 2, 'title' => 'Schedule maintenance for Truck #156', 'description' => 'Regular maintenance due', 'status' => 'pending', 'priority' => 'Medium', 'due_date' => '2025-03-22', 'assignee' => 'John Doe'],
            ['id' => 3, 'title' => 'Review inspection reports', 'description' => 'Review weekly inspection reports and flag any issues', 'status' => 'pending', 'priority' => 'Low', 'due_date' => '2025-03-25', 'assignee' => 'John Doe']
        ];
    }
}

/**
 * Fetch recent alerts
 */
function getAlerts($conn) {
    try {
        $alerts = getAlertsData($conn);
        echo json_encode($alerts);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching alerts: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get alerts data
 */
function getAlertsData($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT id, details AS message, triggered_at AS time
            FROM alerts
            ORDER BY triggered_at DESC
            LIMIT 5
        ");
        $stmt->execute();
        $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Add type field for frontend
        foreach ($alerts as &$alert) {
            $alert['type'] = 'Vehicle Alert'; // Default type
        }

        return $alerts;
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            ['id' => 1, 'type' => 'Vehicle Issue', 'message' => 'Truck #156 reported low brake fluid', 'time' => '10 minutes ago', 'status' => 'active'],
            ['id' => 2, 'type' => 'Inspection Alert', 'message' => '3 vehicles due for inspection today', 'time' => '1 hour ago', 'status' => 'active'],
            ['id' => 3, 'type' => 'Driver Alert', 'message' => 'Driver Sarah Johnson reported delay due to traffic', 'time' => '2 hours ago', 'status' => 'active']
        ];
    }
}

/**
 * Fetch fleet status (counts of vehicles by status)
 */
function getFleetStatus($conn) {
    try {
        $fleetStatus = getFleetStatusData($conn);
        echo json_encode($fleetStatus);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching fleet status: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get fleet status data
 */
function getFleetStatusData($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT status, COUNT(*) as count
            FROM vehicles
            GROUP BY status
        ");
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Map database statuses to frontend keys
        return [
            'available' => $result['ready'] ?? 0,
            'inUse' => $result['in_use'] ?? 0,
            'maintenance' => $result['maintenance_needed'] ?? 0,
            'outOfService' => $result['cleaning'] ?? 0
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'available' => 18,
            'inUse' => 42,
            'maintenance' => 7,
            'outOfService' => 3
        ];
    }
}

/**
 * Fetch inspection statistics
 */
function getInspectionStats($conn) {
    try {
        $inspectionStats = getInspectionStatsData($conn);
        echo json_encode($inspectionStats);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching inspection stats: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get inspection stats data
 */
function getInspectionStatsData($conn) {
    try {
        // Count inspections by status
        $stmt = $conn->prepare("
            SELECT status, COUNT(*) as count
            FROM inspections
            GROUP BY status
        ");
        $stmt->execute();
        $inspectionCounts = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Count scheduled inspections
        $stmt = $conn->prepare("
            SELECT COUNT(*) as scheduled
            FROM inspection_schedules
            WHERE next_inspection > CURDATE()
        ");
        $stmt->execute();
        $scheduled = $stmt->fetchColumn();

        // Map to frontend format
        return [
            'completed' => $inspectionCounts['approved'] ?? 0,
            'pending' => $inspectionCounts['pending'] ?? 0,
            'failed' => $inspectionCounts['flagged'] ?? 0,
            'scheduled' => $scheduled
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'completed' => 78,
            'pending' => 23,
            'failed' => 5,
            'scheduled' => 31
        ];
    }
}

/**
 * Fetch upcoming maintenance (using inspection schedules as proxy)
 */
function getUpcomingMaintenance($conn) {
    try {
        $upcoming = getUpcomingMaintenanceData($conn);
        echo json_encode($upcoming);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching upcoming maintenance: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get upcoming maintenance data
 */
function getUpcomingMaintenanceData($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT v.name AS vehicle, s.next_inspection AS date, 'Inspection' AS type
            FROM inspection_schedules s
            JOIN vehicles v ON s.vehicle_id = v.id
            WHERE s.next_inspection > CURDATE()
            ORDER BY s.next_inspection ASC
            LIMIT 5
        ");
        $stmt->execute();
        $upcoming = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Add ID field for frontend
        foreach ($upcoming as $index => &$item) {
            $item['id'] = $index + 1;
        }

        return $upcoming;
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            ['id' => 1, 'vehicle' => 'Truck #142', 'date' => '2025-03-22', 'type' => 'Oil Change'],
            ['id' => 2, 'vehicle' => 'Van #087', 'date' => '2025-03-23', 'type' => 'Brake Inspection'],
            ['id' => 3, 'vehicle' => 'Truck #265', 'date' => '2025-03-24', 'type' => 'Tire Rotation'],
            ['id' => 4, 'vehicle' => 'SUV #112', 'date' => '2025-03-25', 'type' => 'Full Service']
        ];
    }
}

/**
 * Fetch user profile based on userId parameter
 */
function getUserProfile($conn) {
    try {
        $userId = $_GET['userId'] ?? 7; // Default to Dispatcher One (ID 7)
        $user = getUserProfileData($conn, $userId);
        echo json_encode($user);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching user profile: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get user profile data
 */
function getUserProfileData($conn, $userId = 7) {
    try {
        $stmt = $conn->prepare("
            SELECT name, role, 'https://via.placeholder.com/80' AS avatar,
                   status, '2 minutes ago' AS lastActive
            FROM users
            WHERE id = ?
        ");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($user) {
            return [
                'name' => $user['name'],
                'role' => ucfirst($user['role']),
                'avatar' => $user['avatar'],
                'status' => $user['status'],
                'lastActive' => $user['lastActive']
            ];
        }

        // Return default if user not found
        return [
            'name' => 'John Doe',
            'role' => 'Dispatch Manager',
            'avatar' => 'https://via.placeholder.com/80',
            'status' => 'Active',
            'lastActive' => '2 minutes ago'
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'name' => 'John Doe',
            'role' => 'Dispatch Manager',
            'avatar' => 'https://via.placeholder.com/80',
            'status' => 'Active',
            'lastActive' => '2 minutes ago'
        ];
    }
}

/**
 * Fetch recent inspections
 */
function getRecentInspections($conn) {
    try {
        $inspections = getRecentInspectionsData($conn);
        echo json_encode($inspections);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching recent inspections: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get recent inspections data
 */
function getRecentInspectionsData($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT i.id, v.name AS vehicle, u.name AS inspector,
                   i.created_at AS date, i.status AS result
            FROM inspections i
            JOIN vehicles v ON i.vehicle_id = v.id
            JOIN users u ON i.inspector_id = u.id
            ORDER BY i.created_at DESC
            LIMIT 5
        ");
        $stmt->execute();
        $inspections = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Map status to Passed/Failed for frontend
        foreach ($inspections as &$inspection) {
            $inspection['result'] = $inspection['result'] === 'approved' ? 'passed' : 'failed';
        }

        return $inspections;
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            ['id' => 1, 'vehicle' => 'Truck #156', 'inspector' => 'Alex Johnson', 'date' => '2025-03-19', 'result' => 'passed'],
            ['id' => 2, 'vehicle' => 'Van #092', 'inspector' => 'Sarah Miller', 'date' => '2025-03-19', 'result' => 'failed'],
            ['id' => 3, 'vehicle' => 'Truck #238', 'inspector' => 'David Wilson', 'date' => '2025-03-18', 'result' => 'passed']
        ];
    }
}

/**
 * Helper function to get quick stats data
 */
function getQuickStatsData($conn) {
    try {
        // Get active drivers count
        $stmt1 = $conn->prepare("SELECT COUNT(*) as active_drivers FROM users WHERE role = 'driver' AND status = 'active'");
        $stmt1->execute();
        $activeDrivers = $stmt1->fetch(PDO::FETCH_ASSOC)['active_drivers'] ?? 0;

        // Get pending inspections count
        $stmt2 = $conn->prepare("SELECT COUNT(*) as pending_inspections FROM inspections WHERE status = 'pending'");
        $stmt2->execute();
        $pendingInspections = $stmt2->fetch(PDO::FETCH_ASSOC)['pending_inspections'] ?? 0;

        // Get completed inspections today
        $stmt3 = $conn->prepare("SELECT COUNT(*) as completed_today FROM inspections WHERE status = 'completed' AND DATE(created_at) = CURDATE()");
        $stmt3->execute();
        $completedToday = $stmt3->fetch(PDO::FETCH_ASSOC)['completed_today'] ?? 0;

        // Get average inspection time (hardcoded for now)
        $avgTime = 24;

        return [
            'activeDrivers' => (int)$activeDrivers,
            'pendingInspections' => (int)$pendingInspections,
            'completedToday' => (int)$completedToday,
            'avgInspectionTime' => $avgTime . ' min'
        ];
    } catch (PDOException $e) {
        // Return default values if query fails
        return [
            'activeDrivers' => 12,
            'pendingInspections' => 5,
            'completedToday' => 8,
            'avgInspectionTime' => '24 min'
        ];
    }
}

/**
 * Get fleet locations for the map
 */
function getFleetLocations($conn) {
    try {
        $vehicles = getFleetLocationsData($conn);
        echo json_encode($vehicles);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error fetching fleet locations: ' . $e->getMessage()]);
    }
}

/**
 * Helper function to get fleet locations data
 */
function getFleetLocationsData($conn) {
    try {
        // Get all vehicles with their latest location
        $sql = "SELECT
                    v.id,
                    v.identifier,
                    v.type,
                    v.status,
                    v.license_plate,
                    vl.latitude,
                    vl.longitude,
                    vl.timestamp as location_timestamp,
                    u.name as driver_name,
                    u.id as driver_id
                FROM vehicles v
                LEFT JOIN (
                    SELECT vehicle_id, latitude, longitude, timestamp
                    FROM vehicle_locations vl1
                    WHERE timestamp = (
                        SELECT MAX(timestamp)
                        FROM vehicle_locations vl2
                        WHERE vl1.vehicle_id = vl2.vehicle_id
                    )
                ) vl ON v.id = vl.vehicle_id
                LEFT JOIN users u ON v.assigned_driver_id = u.id
                ORDER BY v.identifier";

        $stmt = $conn->prepare($sql);
        $stmt->execute();

        $vehicles = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // If no vehicles found in the database, return sample data
        if (empty($vehicles)) {
            return getSampleVehicleLocations();
        } else {
            // Format the data for the frontend
            foreach ($vehicles as &$vehicle) {
                // If no location data, assign random coordinates for demo purposes
                if (empty($vehicle['latitude']) || empty($vehicle['longitude'])) {
                    // Generate random coordinates around a central point (e.g., company HQ)
                    // For demo purposes, using coordinates around central US
                    $centerLat = 39.8283; // Central US latitude
                    $centerLng = -98.5795; // Central US longitude
                    $radius = 5; // Radius in degrees

                    $vehicle['latitude'] = $centerLat + (mt_rand(-10000, 10000) / 10000) * $radius;
                    $vehicle['longitude'] = $centerLng + (mt_rand(-10000, 10000) / 10000) * $radius;
                    $vehicle['location_timestamp'] = date('Y-m-d H:i:s');
                }

                // Add a display name for the map
                $vehicle['display_name'] = $vehicle['type'] . ' #' . $vehicle['identifier'];

                // Add a color based on status
                switch ($vehicle['status']) {
                    case 'available':
                    case 'ready':
                        $vehicle['color'] = '#4CAF50'; // Green
                        break;
                    case 'in_use':
                        $vehicle['color'] = '#2196F3'; // Blue
                        break;
                    case 'maintenance':
                    case 'maintenance_needed':
                        $vehicle['color'] = '#9C27B0'; // Purple
                        break;
                    case 'out_of_service':
                    case 'cleaning':
                        $vehicle['color'] = '#F44336'; // Red
                        break;
                    default:
                        $vehicle['color'] = '#757575'; // Grey
                }
            }

            return $vehicles;
        }
    } catch (PDOException $e) {
        // Return sample data on error
        return getSampleVehicleLocations();
    }
}

/**
 * Function to generate sample vehicle locations if no data in database
 */
function getSampleVehicleLocations() {
    // Sample data centered around central US
    $centerLat = 39.8283;
    $centerLng = -98.5795;

    return [
        [
            'id' => 1,
            'identifier' => '101',
            'type' => 'Truck',
            'status' => 'in_use',
            'license_plate' => 'ABC123',
            'latitude' => $centerLat + 0.5,
            'longitude' => $centerLng + 0.8,
            'location_timestamp' => date('Y-m-d H:i:s'),
            'driver_name' => 'John Smith',
            'driver_id' => 10,
            'display_name' => 'Truck #101',
            'color' => '#2196F3'
        ],
        [
            'id' => 2,
            'identifier' => '102',
            'type' => 'Van',
            'status' => 'available',
            'license_plate' => 'DEF456',
            'latitude' => $centerLat - 0.3,
            'longitude' => $centerLng + 0.2,
            'location_timestamp' => date('Y-m-d H:i:s'),
            'driver_name' => null,
            'driver_id' => null,
            'display_name' => 'Van #102',
            'color' => '#4CAF50'
        ],
        [
            'id' => 3,
            'identifier' => '103',
            'type' => 'Truck',
            'status' => 'maintenance',
            'license_plate' => 'GHI789',
            'latitude' => $centerLat + 0.1,
            'longitude' => $centerLng - 0.5,
            'location_timestamp' => date('Y-m-d H:i:s'),
            'driver_name' => null,
            'driver_id' => null,
            'display_name' => 'Truck #103',
            'color' => '#9C27B0'
        ],
        [
            'id' => 4,
            'identifier' => '104',
            'type' => 'SUV',
            'status' => 'out_of_service',
            'license_plate' => 'JKL012',
            'latitude' => $centerLat - 0.7,
            'longitude' => $centerLng - 0.3,
            'location_timestamp' => date('Y-m-d H:i:s'),
            'driver_name' => null,
            'driver_id' => null,
            'display_name' => 'SUV #104',
            'color' => '#F44336'
        ],
        [
            'id' => 5,
            'identifier' => '105',
            'type' => 'Truck',
            'status' => 'in_use',
            'license_plate' => 'MNO345',
            'latitude' => $centerLat + 0.9,
            'longitude' => $centerLng + 0.4,
            'location_timestamp' => date('Y-m-d H:i:s'),
            'driver_name' => 'Sarah Johnson',
            'driver_id' => 11,
            'display_name' => 'Truck #105',
            'color' => '#2196F3'
        ]
    ];
}
?>