<?php
require_once '../db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// --- Authentication Placeholder ---
// In a real app, verify the user's token/session and get their ID and role
// $user_id = $_SESSION['user_id'] ?? null; 
// $user_role = $_SESSION['user_role'] ?? null;
$user_id = 7; // Hardcoding Dispatcher One (ID 7) for now, replace with actual auth
$user_role = 'dispatch'; 
// --- End Authentication Placeholder ---

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            // Get all tasks or a specific task
            $task_id = isset($_GET['id']) ? $_GET['id'] : null;
            
            if ($task_id) {
                // Get a specific task
                $stmt = $pdo->prepare("SELECT 
                                        t.id, 
                                        t.title, 
                                        t.description, 
                                        t.priority, 
                                        t.due_date, 
                                        t.status,
                                        t.assignee_id,
                                        u.name as assignee,
                                        u.user_identifier as assignee_identifier
                                    FROM tasks t
                                    LEFT JOIN users u ON t.assignee_id = u.id
                                    WHERE t.id = :task_id");
                $stmt->bindParam(':task_id', $task_id, PDO::PARAM_INT);
                $stmt->execute();
                
                $task = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($task) {
                    echo json_encode($task);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Task not found']);
                }
            } else {
                // Get all tasks with optional filters
                $status = isset($_GET['status']) ? $_GET['status'] : null;
                
                $sql = "SELECT 
                            t.id, 
                            t.title, 
                            t.description, 
                            t.priority, 
                            t.due_date, 
                            t.status,
                            t.assignee_id,
                            u.name as assignee,
                            u.user_identifier as assignee_identifier
                        FROM tasks t
                        LEFT JOIN users u ON t.assignee_id = u.id";
                
                $params = [];
                
                if ($status) {
                    $sql .= " WHERE t.status = :status";
                    $params[':status'] = $status;
                }
                
                $sql .= " ORDER BY 
                            CASE 
                                WHEN t.priority = 'High' THEN 1
                                WHEN t.priority = 'Medium' THEN 2
                                ELSE 3
                            END,
                            t.due_date ASC";
                
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                
                $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode($tasks);
            }
            break;
            
        case 'POST':
            // Create a new task
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Validate required fields
            if (!isset($data['title']) || empty($data['title'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Title is required']);
                exit;
            }
            
            // Set default values for optional fields
            $description = isset($data['description']) ? $data['description'] : '';
            $priority = isset($data['priority']) ? $data['priority'] : 'Medium';
            $due_date = isset($data['due_date']) ? $data['due_date'] : null;
            $assignee_id = isset($data['assignee_id']) ? $data['assignee_id'] : null;
            
            // If assignee_identifier is provided instead of assignee_id, look up the ID
            if (isset($data['assignee_identifier']) && !$assignee_id) {
                $stmt = $pdo->prepare("SELECT id FROM users WHERE user_identifier = :identifier");
                $stmt->bindParam(':identifier', $data['assignee_identifier'], PDO::PARAM_STR);
                $stmt->execute();
                
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($user) {
                    $assignee_id = $user['id'];
                }
            }
            
            // Insert the new task
            $stmt = $pdo->prepare("INSERT INTO tasks (title, description, priority, due_date, assignee_id, status, created_by) 
                                  VALUES (:title, :description, :priority, :due_date, :assignee_id, 'pending', :created_by)");
            
            $stmt->bindParam(':title', $data['title'], PDO::PARAM_STR);
            $stmt->bindParam(':description', $description, PDO::PARAM_STR);
            $stmt->bindParam(':priority', $priority, PDO::PARAM_STR);
            $stmt->bindParam(':due_date', $due_date, PDO::PARAM_STR);
            $stmt->bindParam(':assignee_id', $assignee_id, $assignee_id ? PDO::PARAM_INT : PDO::PARAM_NULL);
            $stmt->bindParam(':created_by', $user_id, PDO::PARAM_INT);
            
            $stmt->execute();
            
            $task_id = $pdo->lastInsertId();
            
            // Get the newly created task with assignee details
            $stmt = $pdo->prepare("SELECT 
                                    t.id, 
                                    t.title, 
                                    t.description, 
                                    t.priority, 
                                    t.due_date, 
                                    t.status,
                                    t.assignee_id,
                                    u.name as assignee,
                                    u.user_identifier as assignee_identifier
                                FROM tasks t
                                LEFT JOIN users u ON t.assignee_id = u.id
                                WHERE t.id = :task_id");
            $stmt->bindParam(':task_id', $task_id, PDO::PARAM_INT);
            $stmt->execute();
            
            $task = $stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201); // Created
            echo json_encode(['success' => true, 'message' => 'Task created successfully', 'task' => $task]);
            break;
            
        case 'PUT':
            // Update an existing task
            $task_id = isset($_GET['id']) ? $_GET['id'] : null;
            
            if (!$task_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Task ID is required']);
                exit;
            }
            
            // Check if the task exists
            $stmt = $pdo->prepare("SELECT id FROM tasks WHERE id = :task_id");
            $stmt->bindParam(':task_id', $task_id, PDO::PARAM_INT);
            $stmt->execute();
            
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Task not found']);
                exit;
            }
            
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Build the update query dynamically based on provided fields
            $updateFields = [];
            $params = [':task_id' => $task_id];
            
            if (isset($data['title'])) {
                $updateFields[] = "title = :title";
                $params[':title'] = $data['title'];
            }
            
            if (isset($data['description'])) {
                $updateFields[] = "description = :description";
                $params[':description'] = $data['description'];
            }
            
            if (isset($data['priority'])) {
                $updateFields[] = "priority = :priority";
                $params[':priority'] = $data['priority'];
            }
            
            if (isset($data['due_date'])) {
                $updateFields[] = "due_date = :due_date";
                $params[':due_date'] = $data['due_date'];
            }
            
            if (isset($data['status'])) {
                $updateFields[] = "status = :status";
                $params[':status'] = $data['status'];
            }
            
            if (isset($data['assignee_id'])) {
                $updateFields[] = "assignee_id = :assignee_id";
                $params[':assignee_id'] = $data['assignee_id'];
            } elseif (isset($data['assignee_identifier'])) {
                // If assignee_identifier is provided, look up the ID
                $stmt = $pdo->prepare("SELECT id FROM users WHERE user_identifier = :identifier");
                $stmt->bindParam(':identifier', $data['assignee_identifier'], PDO::PARAM_STR);
                $stmt->execute();
                
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($user) {
                    $updateFields[] = "assignee_id = :assignee_id";
                    $params[':assignee_id'] = $user['id'];
                }
            }
            
            if (empty($updateFields)) {
                http_response_code(400);
                echo json_encode(['error' => 'No fields to update']);
                exit;
            }
            
            // Update the task
            $sql = "UPDATE tasks SET " . implode(", ", $updateFields) . " WHERE id = :task_id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            
            // Get the updated task
            $stmt = $pdo->prepare("SELECT 
                                    t.id, 
                                    t.title, 
                                    t.description, 
                                    t.priority, 
                                    t.due_date, 
                                    t.status,
                                    t.assignee_id,
                                    u.name as assignee,
                                    u.user_identifier as assignee_identifier
                                FROM tasks t
                                LEFT JOIN users u ON t.assignee_id = u.id
                                WHERE t.id = :task_id");
            $stmt->bindParam(':task_id', $task_id, PDO::PARAM_INT);
            $stmt->execute();
            
            $task = $stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'message' => 'Task updated successfully', 'task' => $task]);
            break;
            
        case 'DELETE':
            // Delete a task
            $task_id = isset($_GET['id']) ? $_GET['id'] : null;
            
            if (!$task_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Task ID is required']);
                exit;
            }
            
            // Check if the task exists
            $stmt = $pdo->prepare("SELECT id FROM tasks WHERE id = :task_id");
            $stmt->bindParam(':task_id', $task_id, PDO::PARAM_INT);
            $stmt->execute();
            
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Task not found']);
                exit;
            }
            
            // Delete the task
            $stmt = $pdo->prepare("DELETE FROM tasks WHERE id = :task_id");
            $stmt->bindParam(':task_id', $task_id, PDO::PARAM_INT);
            $stmt->execute();
            
            echo json_encode(['success' => true, 'message' => 'Task deleted successfully']);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}
?>
