<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Database connection
require_once 'db_connect.php';

// Get request body
$data = json_decode(file_get_contents('php://input'), true);

// Check if required fields are provided
if (!isset($data['username']) || !isset($data['pin'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Username and PIN are required']);
    exit;
}

$username = $data['username'];
$pin = $data['pin']; // This should be already hashed from the frontend

try {
    // Find user by username
    $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
    $stmt->execute([$username]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // User not found, try with email
        $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
    }

    if ($user) {
        // Verify PIN - handle both direct comparison (for SHA-256 hashed PINs) and PHP's password_verify
        $authenticated = false;
        
        // Direct comparison for SHA-256 hashed PINs from frontend
        if ($pin === $user['pin']) {
            $authenticated = true;
        }
        // For backward compatibility with PHP's password_hash
        else if (password_verify($pin, $user['pin'])) {
            $authenticated = true;
        }

        if ($authenticated) {
            // Reset failed login attempts
            $updateAttempts = $pdo->prepare("UPDATE users SET failed_login_attempts = 0, last_login = NOW() WHERE id = ?");
            $updateAttempts->execute([$user['id']]);
            
            // Generate a new token
            $token = bin2hex(random_bytes(32));
            
            // Update user's token
            $updateToken = $pdo->prepare("UPDATE users SET token = ? WHERE id = ?");
            $updateToken->execute([$token, $user['id']]);

            // Prepare user data for response
            $userData = [
                'id' => $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'full_name' => $user['full_name'],
                'role' => $user['role'],
                'company' => $user['company'],
                'created_at' => $user['created_at']
            ];

            // Log successful login
            logActivity($pdo, $user['id'], 'login', 'Successful login');

            // Return success response
            echo json_encode([
                'success' => true, 
                'message' => 'Login successful',
                'user' => $userData,
                'token' => $token
            ]);
        } else {
            // Increment failed login attempts
            $attempts = $user['failed_login_attempts'] + 1;
            $updateAttempts = $pdo->prepare("UPDATE users SET failed_login_attempts = ? WHERE id = ?");
            $updateAttempts->execute([$attempts, $user['id']]);
            
            // Log failed login attempt
            logActivity($pdo, null, 'login_failed', "Failed login attempt for username: $username");

            // Return error response
            echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
        }
    } else {
        // Log failed login attempt
        logActivity($pdo, null, 'login_failed', "Failed login attempt for username: $username (user not found)");
        
        // Return error response
        echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
    }
} catch (PDOException $e) {
    // Log error
    error_log("Login error: " . $e->getMessage());
    
    // Return error response
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error']);
}

// Helper function to log activity
function logActivity($pdo, $userId, $activityType, $details) {
    try {
        // Create activity_logs table if it doesn't exist
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS activity_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT,
                activity_type VARCHAR(50) NOT NULL,
                details TEXT,
                ip_address VARCHAR(45),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX (user_id),
                INDEX (activity_type)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ");
        
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs (user_id, activity_type, details, ip_address) 
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([$userId, $activityType, $details, $_SERVER['REMOTE_ADDR']]);
    } catch (PDOException $e) {
        // Just log the error but don't stop the login process
        error_log("Activity logging error: " . $e->getMessage());
    }
}
?>