<?php
require_once 'db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

echo "<h1>Populate Inspections Table</h1>";

try {
    // First, check if we have vehicles in the database
    $checkVehiclesStmt = $pdo->query("SELECT COUNT(*) FROM vehicles");
    $vehicleCount = $checkVehiclesStmt->fetchColumn();
    
    if ($vehicleCount == 0) {
        // Create some sample vehicles first
        echo "<p>No vehicles found. Creating sample vehicles...</p>";
        
        $vehicleData = [
            ['Tractor 101', 'tractor', 'active', 1, 'TR101'],
            ['Harvester 202', 'harvester', 'active', 1, 'HV202'],
            ['Sprayer 303', 'sprayer', 'maintenance', 1, 'SP303'],
            ['Tiller 404', 'tiller', 'active', 1, 'TL404'],
            ['Combine 505', 'combine', 'inactive', 1, 'CB505']
        ];
        
        $insertVehicleStmt = $pdo->prepare("
            INSERT INTO vehicles (name, type, status, yard_id, vehicle_identifier) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($vehicleData as $vehicle) {
            $insertVehicleStmt->execute($vehicle);
        }
        
        echo "<p>Created " . count($vehicleData) . " sample vehicles.</p>";
    } else {
        echo "<p>Found $vehicleCount vehicles in the database.</p>";
    }
    
    // Now get vehicle IDs for creating inspections
    $vehiclesStmt = $pdo->query("SELECT id FROM vehicles LIMIT 5");
    $vehicleIds = $vehiclesStmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (count($vehicleIds) == 0) {
        echo "<p>Error: No vehicle IDs found even after attempting to create vehicles.</p>";
        exit;
    }
    
    // Check if we already have inspections
    $checkInspectionsStmt = $pdo->query("SELECT COUNT(*) FROM inspections");
    $inspectionCount = $checkInspectionsStmt->fetchColumn();
    
    if ($inspectionCount > 0) {
        echo "<p>There are already $inspectionCount inspections in the database. Skipping creation of sample inspections.</p>";
    } else {
        // Create sample inspections
        echo "<p>Creating sample inspections...</p>";
        
        $statuses = ['pending', 'approved', 'flagged'];
        $inspectionData = [];
        
        // Create 10 sample inspections
        for ($i = 0; $i < 10; $i++) {
            $vehicleId = $vehicleIds[array_rand($vehicleIds)];
            $status = $statuses[array_rand($statuses)];
            $date = date('Y-m-d', strtotime("-" . rand(0, 30) . " days"));
            
            $inspectionData[] = [$vehicleId, $date, $status];
        }
        
        $insertInspectionStmt = $pdo->prepare("
            INSERT INTO inspections (vehicle_id, inspection_date, status) 
            VALUES (?, ?, ?)
        ");
        
        foreach ($inspectionData as $inspection) {
            $insertInspectionStmt->execute($inspection);
        }
        
        echo "<p>Created " . count($inspectionData) . " sample inspections.</p>";
    }
    
    // Display all inspections with vehicle information
    $inspectionsStmt = $pdo->query("
        SELECT i.id, v.vehicle_identifier, i.inspection_date, i.status 
        FROM inspections i
        JOIN vehicles v ON i.vehicle_id = v.id
        ORDER BY i.inspection_date DESC
    ");
    
    $inspections = $inspectionsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<h2>Current Inspections (" . count($inspections) . ")</h2>";
    
    if (count($inspections) > 0) {
        echo "<table border='1' cellpadding='5'>";
        echo "<tr><th>ID</th><th>Vehicle</th><th>Date</th><th>Status</th></tr>";
        
        foreach ($inspections as $inspection) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($inspection['id']) . "</td>";
            echo "<td>" . htmlspecialchars($inspection['vehicle_identifier']) . "</td>";
            echo "<td>" . htmlspecialchars($inspection['inspection_date']) . "</td>";
            echo "<td>" . htmlspecialchars($inspection['status']) . "</td>";
            echo "</tr>";
        }
        
        echo "</table>";
    } else {
        echo "<p>No inspections found in the database.</p>";
    }
    
    echo "<p><a href='subpages/yard/inspections.php'>View JSON API Response</a></p>";
    
} catch (PDOException $e) {
    echo "<h2>Error</h2>";
    echo "<p>Database error: " . htmlspecialchars($e->getMessage()) . "</p>";
}
?>
