<?php
require_once 'db_connect.php';
header("Access-Control-Allow-Origin: *");
header("Content-Type: text/html");

echo "<h1>Populate Vehicles Table</h1>";

try {
    // First, check if we have vehicle types in the database
    $checkTypesStmt = $pdo->query("SELECT COUNT(*) FROM vehicle_types");
    $typeCount = $checkTypesStmt->fetchColumn();
    
    if ($typeCount == 0) {
        // Create some sample vehicle types first
        echo "<p>No vehicle types found. Creating sample vehicle types...</p>";
        
        $typeData = [
            ['Tractor', 'Agricultural tractor'],
            ['Harvester', 'Crop harvesting machine'],
            ['Sprayer', 'Crop spraying equipment'],
            ['Tiller', 'Soil preparation equipment'],
            ['Combine', 'Harvesting and threshing machine']
        ];
        
        $insertTypeStmt = $pdo->prepare("
            INSERT INTO vehicle_types (name, description) 
            VALUES (?, ?)
        ");
        
        foreach ($typeData as $type) {
            $insertTypeStmt->execute($type);
        }
        
        echo "<p>Created " . count($typeData) . " sample vehicle types.</p>";
    } else {
        echo "<p>Found $typeCount vehicle types in the database.</p>";
    }
    
    // Check if we have yards in the database
    $checkYardsStmt = $pdo->query("SELECT COUNT(*) FROM yards");
    $yardCount = $checkYardsStmt->fetchColumn();
    
    if ($yardCount == 0) {
        // Create some sample yards first
        echo "<p>No yards found. Creating sample yards...</p>";
        
        $yardData = [
            ['Main Yard', 'Primary equipment yard', 1],
            ['North Yard', 'Northern region yard', 1],
            ['South Yard', 'Southern region yard', 1]
        ];
        
        $insertYardStmt = $pdo->prepare("
            INSERT INTO yards (name, description, region_id) 
            VALUES (?, ?, ?)
        ");
        
        foreach ($yardData as $yard) {
            $insertYardStmt->execute($yard);
        }
        
        echo "<p>Created " . count($yardData) . " sample yards.</p>";
    } else {
        echo "<p>Found $yardCount yards in the database.</p>";
    }
    
    // Now get vehicle type IDs and yard IDs for creating vehicles
    $typesStmt = $pdo->query("SELECT id FROM vehicle_types");
    $typeIds = $typesStmt->fetchAll(PDO::FETCH_COLUMN);
    
    $yardsStmt = $pdo->query("SELECT id FROM yards");
    $yardIds = $yardsStmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (count($typeIds) == 0 || count($yardIds) == 0) {
        echo "<p>Error: No vehicle types or yards found even after attempting to create them.</p>";
        exit;
    }
    
    // Check if we already have vehicles
    $checkVehiclesStmt = $pdo->query("SELECT COUNT(*) FROM vehicles");
    $vehicleCount = $checkVehiclesStmt->fetchColumn();
    
    if ($vehicleCount > 0) {
        echo "<p>There are already $vehicleCount vehicles in the database. Skipping creation of sample vehicles.</p>";
    } else {
        // Create sample vehicles
        echo "<p>Creating sample vehicles...</p>";
        
        $statuses = ['Operational', 'Maintenance', 'Out of Service', 'On Route', 'Ready', 'In Use', 'Cleaning'];
        $locations = ['Field A', 'Field B', 'Warehouse', 'Maintenance Bay', 'Yard'];
        $vehicleData = [];
        
        // Create 15 sample vehicles
        for ($i = 1; $i <= 15; $i++) {
            $typeId = $typeIds[array_rand($typeIds)];
            $yardId = $yardIds[array_rand($yardIds)];
            $status = $statuses[array_rand($statuses)];
            $location = $locations[array_rand($locations)];
            $identifier = 'VEH-' . str_pad($i, 3, '0', STR_PAD_LEFT);
            $maintenanceDue = date('Y-m-d', strtotime('+' . rand(1, 90) . ' days'));
            
            $vehicleData[] = [$identifier, $typeId, $status, $location, $yardId, $maintenanceDue];
        }
        
        $insertVehicleStmt = $pdo->prepare("
            INSERT INTO vehicles (vehicle_identifier, vehicle_type_id, status, location_text, yard_id, maintenance_due_date) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        foreach ($vehicleData as $vehicle) {
            $insertVehicleStmt->execute($vehicle);
        }
        
        echo "<p>Created " . count($vehicleData) . " sample vehicles.</p>";
    }
    
    // Create maintenance records for vehicles
    $checkMaintenanceStmt = $pdo->query("SELECT COUNT(*) FROM maintenance");
    $maintenanceCount = $checkMaintenanceStmt->fetchColumn();
    
    if ($maintenanceCount > 0) {
        echo "<p>There are already $maintenanceCount maintenance records in the database. Skipping creation of sample maintenance records.</p>";
    } else {
        // Get all vehicle IDs
        $vehiclesStmt = $pdo->query("SELECT id FROM vehicles");
        $vehicleIds = $vehiclesStmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (count($vehicleIds) > 0) {
            echo "<p>Creating sample maintenance records...</p>";
            
            $serviceTypes = ['oil_change', 'tire_rotation', 'engine_check', 'brake_service', 'general_maintenance'];
            $maintenanceData = [];
            
            // Create 30 sample maintenance records
            for ($i = 0; $i < 30; $i++) {
                $vehicleId = $vehicleIds[array_rand($vehicleIds)];
                $serviceType = $serviceTypes[array_rand($serviceTypes)];
                $serviceDate = date('Y-m-d', strtotime('-' . rand(1, 180) . ' days'));
                $notes = 'Routine ' . str_replace('_', ' ', $serviceType);
                
                $maintenanceData[] = [$vehicleId, $serviceType, $serviceDate, $notes];
            }
            
            $insertMaintenanceStmt = $pdo->prepare("
                INSERT INTO maintenance (vehicle_id, service_type, service_date, notes) 
                VALUES (?, ?, ?, ?)
            ");
            
            foreach ($maintenanceData as $maintenance) {
                $insertMaintenanceStmt->execute($maintenance);
            }
            
            echo "<p>Created " . count($maintenanceData) . " sample maintenance records.</p>";
        }
    }
    
    // Display all vehicles with their information
    $vehiclesStmt = $pdo->query("
        SELECT 
            v.id, 
            v.vehicle_identifier, 
            vt.name AS vehicle_type, 
            v.status, 
            v.location_text, 
            y.name AS yard_name,
            v.maintenance_due_date
        FROM vehicles v
        JOIN vehicle_types vt ON v.vehicle_type_id = vt.id
        JOIN yards y ON v.yard_id = y.id
        ORDER BY v.id
    ");
    
    $vehicles = $vehiclesStmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<h2>Current Vehicles (" . count($vehicles) . ")</h2>";
    
    if (count($vehicles) > 0) {
        echo "<table border='1' cellpadding='5'>";
        echo "<tr><th>ID</th><th>Identifier</th><th>Type</th><th>Status</th><th>Location</th><th>Yard</th><th>Maintenance Due</th></tr>";
        
        foreach ($vehicles as $vehicle) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($vehicle['id']) . "</td>";
            echo "<td>" . htmlspecialchars($vehicle['vehicle_identifier']) . "</td>";
            echo "<td>" . htmlspecialchars($vehicle['vehicle_type']) . "</td>";
            echo "<td>" . htmlspecialchars($vehicle['status']) . "</td>";
            echo "<td>" . htmlspecialchars($vehicle['location_text']) . "</td>";
            echo "<td>" . htmlspecialchars($vehicle['yard_name']) . "</td>";
            echo "<td>" . htmlspecialchars($vehicle['maintenance_due_date']) . "</td>";
            echo "</tr>";
        }
        
        echo "</table>";
    } else {
        echo "<p>No vehicles found in the database.</p>";
    }
    
    echo "<p><a href='subpages/yard/vehicle_status.php'>View JSON API Response</a></p>";
    
} catch (PDOException $e) {
    echo "<h2>Error</h2>";
    echo "<p>Database error: " . htmlspecialchars($e->getMessage()) . "</p>";
}
?>
