<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../db_connect.php';

// Check if database connection was successful
if (!isset($pdo) || $pdo === null) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit();
}

try {
    // Function to check if a table exists
    function tableExists($tableName, $pdo) {
        $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
        $stmt->execute([$tableName]);
        return $stmt->rowCount() > 0;
    }
    
    // Function to count rows in a table
    function countRows($tableName, $pdo) {
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM $tableName");
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['count'];
    }
    
    $results = [];
    
    // 1. Check and seed yards table
    if (!tableExists('yards', $pdo)) {
        // Create yards table
        $pdo->exec("CREATE TABLE yards (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL,
            region_id INT,
            location VARCHAR(255),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $results['yards'] = 'Table created';
    }
    
    // Seed yards data if empty
    if (countRows('yards', $pdo) === 0) {
        $yardNames = ['Yard Alpha', 'Yard Bravo', 'Yard Charlie', 'Yard Delta', 'Yard Echo'];
        $locations = ['North Region', 'South Region', 'East Region', 'West Region', 'Central Region'];
        $yardCount = 0;
        
        for ($i = 0; $i < count($yardNames); $i++) {
            $regionId = $i % 3 + 1; // Assign to regions 1, 2, or 3
            
            $stmt = $pdo->prepare("INSERT INTO yards (name, region_id, location) VALUES (?, ?, ?)");
            $stmt->execute([$yardNames[$i], $regionId, $locations[$i]]);
            $yardCount++;
        }
        
        $results['yards'] = "Seeded with $yardCount records";
    } else {
        $results['yards'] = "Table already has data";
    }
    
    // 2. Ensure we have some vehicles with yard_id
    if (!tableExists('vehicles', $pdo)) {
        // Create vehicles table if it doesn't exist
        $pdo->exec("CREATE TABLE vehicles (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL,
            type VARCHAR(50) NOT NULL,
            status VARCHAR(50) NOT NULL,
            yard_id INT,
            year INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $results['vehicles'] = 'Table created';
    } else {
        // Check if yard_id column exists
        $stmt = $pdo->prepare("SHOW COLUMNS FROM vehicles LIKE 'yard_id'");
        $stmt->execute();
        if ($stmt->rowCount() === 0) {
            // Add yard_id column
            $pdo->exec("ALTER TABLE vehicles ADD COLUMN yard_id INT AFTER status");
            $results['vehicles_alter'] = 'Added yard_id column';
        }
    }
    
    // Seed vehicles data if empty
    if (countRows('vehicles', $pdo) === 0) {
        $vehicleTypes = ['Tractor', 'Harvester', 'Utility Vehicle', 'Truck', 'Trailer'];
        $statuses = ['ready', 'in_use', 'maintenance_needed', 'out_of_service'];
        $vehicleCount = 0;
        
        // Create 100 vehicles with different types and statuses
        for ($i = 1; $i <= 100; $i++) {
            $type = $vehicleTypes[array_rand($vehicleTypes)];
            
            // Weighted status distribution
            $rand = rand(1, 100);
            if ($rand <= 50) {
                $status = 'ready';
            } elseif ($rand <= 80) {
                $status = 'in_use';
            } elseif ($rand <= 95) {
                $status = 'maintenance_needed';
            } else {
                $status = 'out_of_service';
            }
            
            $yardId = rand(1, 5); // Assign to one of the 5 yards
            $year = rand(2015, 2023);
            $name = "$type-" . str_pad($i, 3, '0', STR_PAD_LEFT);
            
            $stmt = $pdo->prepare("INSERT INTO vehicles (name, type, status, yard_id, year) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$name, $type, $status, $yardId, $year]);
            $vehicleCount++;
        }
        
        $results['vehicles'] = "Seeded with $vehicleCount records";
    } else {
        // Update yard_id for existing vehicles if not set
        $stmt = $pdo->prepare("UPDATE vehicles SET yard_id = (id % 5) + 1 WHERE yard_id IS NULL OR yard_id = 0");
        $stmt->execute();
        $updatedCount = $stmt->rowCount();
        
        if ($updatedCount > 0) {
            $results['vehicles_update'] = "Updated yard_id for $updatedCount vehicles";
        } else {
            $results['vehicles'] = "Table already has data with yard_id";
        }
    }
    
    // 3. Ensure we have inspections with proper statuses
    if (!tableExists('inspections', $pdo)) {
        // Create inspections table
        $pdo->exec("CREATE TABLE inspections (
            id INT AUTO_INCREMENT PRIMARY KEY,
            vehicle_id INT NOT NULL,
            inspector_id INT,
            status VARCHAR(50) NOT NULL,
            inspection_date DATE NOT NULL,
            notes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $results['inspections'] = 'Table created';
    }
    
    // Seed inspections data if empty
    if (countRows('inspections', $pdo) === 0) {
        $statuses = ['passed', 'failed', 'pending', 'in_progress'];
        $inspectionCount = 0;
        
        // Generate data for the last 6 months
        for ($month = 5; $month >= 0; $month--) {
            $daysInMonth = date('t', strtotime("-$month months"));
            $inspectionsPerMonth = rand(20, 40);
            
            for ($i = 0; $i < $inspectionsPerMonth; $i++) {
                $day = rand(1, $daysInMonth);
                $date = date('Y-m-d', strtotime("-$month months -" . ($daysInMonth - $day) . " days"));
                
                // Weighted status distribution (more passes than fails)
                $rand = rand(1, 100);
                if ($rand <= 70) {
                    $status = 'passed';
                } elseif ($rand <= 85) {
                    $status = 'failed';
                } elseif ($rand <= 95) {
                    $status = 'pending';
                } else {
                    $status = 'in_progress';
                }
                
                $vehicleId = rand(1, 100);
                $inspectorId = rand(1, 5);
                
                $stmt = $pdo->prepare("INSERT INTO inspections (vehicle_id, inspector_id, status, inspection_date, notes) 
                                      VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$vehicleId, $inspectorId, $status, $date, "Inspection notes for vehicle $vehicleId"]);
                $inspectionCount++;
            }
        }
        
        $results['inspections'] = "Seeded with $inspectionCount records";
    } else {
        $results['inspections'] = "Table already has data";
    }
    
    // 4. Ensure we have maintenance records
    if (!tableExists('maintenance', $pdo)) {
        // Create maintenance table
        $pdo->exec("CREATE TABLE maintenance (
            id INT AUTO_INCREMENT PRIMARY KEY,
            vehicle_id INT NOT NULL,
            type VARCHAR(50) NOT NULL,
            status VARCHAR(50) NOT NULL,
            start_date DATE NOT NULL,
            end_date DATE,
            cost DECIMAL(10,2),
            notes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $results['maintenance'] = 'Table created';
    }
    
    // Seed maintenance data if empty
    if (countRows('maintenance', $pdo) === 0) {
        $maintenanceTypes = ['Oil Change', 'Tire Rotation', 'Engine Repair', 'Brake Service', 'Electrical', 'Transmission', 'Cooling System', 'Fuel System', 'Suspension', 'Exhaust'];
        $statuses = ['completed', 'in_progress', 'scheduled'];
        $maintenanceCount = 0;
        
        // Create maintenance records for the last 6 months
        for ($month = 5; $month >= 0; $month--) {
            $recordsPerMonth = rand(15, 30);
            
            for ($i = 0; $i < $recordsPerMonth; $i++) {
                $vehicleId = rand(1, 100);
                $type = $maintenanceTypes[array_rand($maintenanceTypes)];
                $status = $statuses[array_rand($statuses)];
                
                $startDate = date('Y-m-d', strtotime("-$month months -" . rand(0, 30) . " days"));
                $endDate = ($status === 'completed') ? date('Y-m-d', strtotime($startDate . " +" . rand(1, 5) . " days")) : null;
                $cost = rand(50, 2000);
                
                $stmt = $pdo->prepare("INSERT INTO maintenance (vehicle_id, type, status, start_date, end_date, cost, notes) 
                                      VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$vehicleId, $type, $status, $startDate, $endDate, $cost, "Maintenance notes for $type"]);
                $maintenanceCount++;
            }
        }
        
        $results['maintenance'] = "Seeded with $maintenanceCount records";
    } else {
        $results['maintenance'] = "Table already has data";
    }
    
    // Return results
    echo json_encode(['success' => true, 'results' => $results]);
    
} catch (Exception $e) {
    // Log the error server-side
    error_log('Error in seed_compliance_data.php: ' . $e->getMessage());
    
    // Return a clean error message to the client
    http_response_code(500);
    echo json_encode(['error' => 'Server error occurred: ' . $e->getMessage()]);
}

// Close database connection
$pdo = null;
?>
