<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json; charset=UTF-8");

require_once 'db_connect.php';

// Get JSON input
$data = json_decode(file_get_contents("php://input"), true);

// Validate required fields
$requiredFields = ['username', 'email', 'pin', 'full_name', 'role', 'region_id'];
$missingFields = [];

foreach ($requiredFields as $field) {
    if (!isset($data[$field]) || empty($data[$field])) {
        $missingFields[] = $field;
    }
}

if (!empty($missingFields)) {
    http_response_code(400);
    echo json_encode([
        'success' => false, 
        'message' => 'Required fields missing: ' . implode(', ', $missingFields)
    ]);
    exit;
}

try {
    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$data['email']]);
    
    if ($stmt->rowCount() > 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Email already registered'
        ]);
        exit;
    }

    // Hash the PIN
    $hashedPin = hash('sha256', $data['pin']);

    // Insert new user
    $stmt = $pdo->prepare("
        INSERT INTO users (
            username,
            email,
            pin,
            name,
            role,
            region_id,
            status,
            created_at
        ) VALUES (
            :username,
            :email,
            :pin,
            :name,
            :role,
            :region_id,
            :status,
            NOW()
        )
    ");

    $result = $stmt->execute([
        ':username' => $data['username'],
        ':email' => $data['email'],
        ':pin' => $hashedPin,
        ':name' => $data['full_name'],
        ':role' => $data['role'],
        ':region_id' => $data['region_id'],
        ':status' => 'active'
    ]);

    if ($result) {
        $userId = $pdo->lastInsertId();
        echo json_encode([
            'success' => true,
            'message' => 'User registered successfully',
            'user_id' => $userId
        ]);
    } else {
        throw new Exception('Failed to insert user');
    }

} catch (PDOException $e) {
    error_log("Registration error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database error occurred'
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
