<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Database connection
require_once 'db_connect.php';

// Get request body
$data = json_decode(file_get_contents('php://input'), true);

// Check if required fields are provided
// Map frontend 'username' to 'user_identifier', 'full_name' to 'name'
// 'pin' is now plain text
if (!isset($data['username']) || !isset($data['email']) || !isset($data['pin']) || !isset($data['full_name']) || !isset($data['role'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Required fields are missing']);
    exit;
}

$userIdentifier = $data['username']; // Frontend 'username' maps to 'user_identifier'
$email = $data['email'];
$plainPin = $data['pin']; // This is now the plain PIN
$name = $data['full_name']; // Frontend 'full_name' maps to 'name'
$role = $data['role'];
// 'company' and 'phone' are not in the 'users' table schema found
// $company = isset($data['company']) ? $data['company'] : null;
// $phone = isset($data['phone']) ? $data['phone'] : null;

// Validate PIN length (assuming it should still be 4 digits)
if (strlen($plainPin) !== 4 || !ctype_digit($plainPin)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'PIN must be exactly 4 digits']);
    exit;
}

try {
    // Check if email already exists in the 'users' table
    // Removed check for user_identifier as the column doesn't exist
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);

    if ($stmt->rowCount() > 0) {
        echo json_encode(['success' => false, 'message' => 'Email already exists']);
        exit;
    }

    // Hash the plain PIN using password_hash()
    //$hashedPassword = password_hash($plainPin, PASSWORD_DEFAULT);

    // Hash the PIN using SHA-256
    $hashedPassword =  hash('sha256', $plainPin);

    // Insert new user into the 'users' table
    // Removed user_identifier from INSERT statement
    $stmt = $pdo->prepare("
        INSERT INTO users (email, password, name, username, pin, role, status)
        VALUES (?, ?, ?, ?, ?, ?, 'active')
    ");
    // Note: Removed company, phone, user_identifier. Added password (hashed), name.
    // Assuming region_id and yard_id are optional or set later.

    $result = $stmt->execute([$email, $hashedPassword, $name, $userIdentifier, $hashedPassword, $role]);

    if ($result) {
        $userId = $pdo->lastInsertId();
        echo json_encode(['success' => true, 'message' => 'Registration successful', 'user_id' => $userId]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Registration failed']);
    }
} catch (PDOException $e) {
    // Log error
    error_log("Registration error: " . $e->getMessage());
    
    // Return error response
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
?>
