<?php
// Add these headers at the very top of the file
header("Access-Control-Allow-Origin: http://localhost:3000");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Credentials: true");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once '../db_connect.php';

try {
    // Get JSON input
    $data = json_decode(file_get_contents("php://input"), true);

    // Validate required fields
    $requiredFields = ['username', 'email', 'pin', 'role'];
    $missingFields = [];

    foreach ($requiredFields as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            $missingFields[] = $field;
        }
    }

    if (!empty($missingFields)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Required fields missing: ' . implode(', ', $missingFields)
        ]);
        exit;
    }

    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$data['email']]);

    if ($stmt->rowCount() > 0) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Email already exists'
        ]);
        exit;
    }

    // Hash the PIN
    $hashedPin = password_hash($data['pin'], PASSWORD_DEFAULT);

    // Insert new user
    $stmt = $pdo->prepare("
        INSERT INTO users (
            username,
            email,
            pin,
            role,
            status,
            created_at,
            created_by
        ) VALUES (
            :username,
            :email,
            :pin,
            :role,
            :status,
            NOW(),
            :created_by
        )
    ");

    $result = $stmt->execute([
        ':username' => $data['username'],
        ':email' => $data['email'],
        ':pin' => $hashedPin,
        ':role' => $data['role'],
        ':status' => 'active',
        ':created_by' => $data['created_by'] ?? null
    ]);

    if ($result) {
        $userId = $pdo->lastInsertId();
        echo json_encode([
            'success' => true,
            'message' => 'User registered successfully',
            'user_id' => $userId
        ]);
    } else {
        throw new Exception('Failed to insert user');
    }

} catch (PDOException $e) {
    error_log("Registration error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database error occurred'
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>