<?php
require_once 'api.php';

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        $userId = validateToken();
        
        $stmt = $pdo->prepare("
            SELECT 
                st.id,
                st.title,
                st.description,
                st.status,
                st.priority,
                st.created_at,
                st.updated_at,
                st.category,
                COALESCE(r.message, '') as latest_response,
                COALESCE(r.created_at, '') as response_date
            FROM support_tickets st
            LEFT JOIN (
                SELECT 
                    ticket_id,
                    message,
                    created_at,
                    ROW_NUMBER() OVER (PARTITION BY ticket_id ORDER BY created_at DESC) as rn
                FROM ticket_responses
            ) r ON r.ticket_id = st.id AND r.rn = 1
            WHERE st.user_id = ?
            ORDER BY st.created_at DESC
        ");
        
        $stmt->execute([$userId]);
        $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $tickets
        ]);
        
    } catch (Exception $e) {
        handleError($e->getMessage());
    }
}

else if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $userId = validateToken();
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        if (!isset($data['title']) || !isset($data['description']) || !isset($data['category'])) {
            handleError('Missing required fields', 400);
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO support_tickets 
            (user_id, title, description, category, priority, status, created_at)
            VALUES (?, ?, ?, ?, ?, 'Open', NOW())
        ");
        
        $stmt->execute([
            $userId,
            $data['title'],
            $data['description'],
            $data['category'],
            $data['priority'] ?? 'Medium'
        ]);
        
        $ticketId = $pdo->lastInsertId();
        
        // Send notification to support team (implement your notification logic here)
        // notifySupportTeam($ticketId);
        
        echo json_encode([
            'success' => true,
            'message' => 'Support ticket created successfully',
            'id' => $ticketId
        ]);
        
    } catch (Exception $e) {
        handleError($e->getMessage());
    }
}

else if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    try {
        $userId = validateToken();
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id']) || !isset($data['message'])) {
            handleError('Missing required fields', 400);
        }
        
        // First verify the ticket belongs to the user
        $stmt = $pdo->prepare("
            SELECT id FROM support_tickets 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$data['id'], $userId]);
        
        if (!$stmt->fetch()) {
            handleError('Ticket not found or access denied', 403);
        }
        
        // Add the response
        $stmt = $pdo->prepare("
            INSERT INTO ticket_responses 
            (ticket_id, user_id, message, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $data['id'],
            $userId,
            $data['message']
        ]);
        
        // Update the ticket's updated_at timestamp
        $stmt = $pdo->prepare("
            UPDATE support_tickets 
            SET updated_at = NOW()
            WHERE id = ?
        ");
        
        $stmt->execute([$data['id']]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Response added successfully'
        ]);
        
    } catch (Exception $e) {
        handleError($e->getMessage());
    }
}

else {
    handleError('Method not allowed', 405);
}

function handleError($message, $code = 500) {
    http_response_code($code);
    echo json_encode([
        'success' => false,
        'error' => $message
    ]);
    exit;
}
?>