<?php
header("Access-Control-Allow-Origin: *"); // Allow requests from any origin (adjust for production)
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, OPTIONS"); // Allow POST and OPTIONS methods (DELETE might be more semantic, but POST is often used for simplicity)
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request (pre-flight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../../db_connect.php'; // Adjust path as needed

// Get posted data (expecting JSON with just 'id')
$data = json_decode(file_get_contents("php://input"));

// Basic validation
if (!isset($data->id) || empty($data->id)) {
    http_response_code(400); // Bad Request
    echo json_encode(array("error" => "User ID not provided."));
    exit();
}

// Sanitize input
$id = filter_var($data->id, FILTER_SANITIZE_NUMBER_INT);

if (!$id || $id <= 0) { // Ensure ID is a positive integer
    http_response_code(400); // Bad Request
    echo json_encode(array("error" => "Invalid User ID provided."));
    exit();
}

// --- Optional: Add check to prevent deleting critical users (e.g., the main owner) ---
// Example: Check if the user being deleted is the currently logged-in user or a specific admin ID
// if ($id == 1) { // Assuming ID 1 is a super admin or owner that shouldn't be deleted
//     http_response_code(403); // Forbidden
//     echo json_encode(array("error" => "This user cannot be deleted."));
//     exit();
// }
// --- End Optional Check ---


try {
    // $pdo is created in db_connect.php
    if (!isset($pdo) || !$pdo instanceof PDO) {
        // This condition might be hit if db_connect.php failed silently
        throw new Exception("Database connection object not available.");
    }

    // Prepare delete statement
    $sql = "DELETE FROM users WHERE id = :id";
    $stmt = $pdo->prepare($sql);

    // Bind parameter
    $stmt->bindParam(':id', $id, PDO::PARAM_INT);

    // Execute the query
    if ($stmt->execute()) {
        if ($stmt->rowCount() > 0) {
            http_response_code(200); // OK
            echo json_encode(array("message" => "User deleted successfully."));
        } else {
            // ID was valid format, but no user found with that ID
            http_response_code(404); // Not Found
            echo json_encode(array("error" => "User not found with the specified ID."));
        }
    } else {
        // Check for foreign key constraint errors specifically if possible
        // PDO errorInfo can be checked, e.g., $stmt->errorInfo()[1] == 1451 for constraint violation
        if ($stmt->errorInfo()[1] == 1451) {
             http_response_code(409); // Conflict
             echo json_encode(array("error" => "Cannot delete user. They may be referenced in other records (e.g., inspections, tasks). Consider deactivating instead."));
        } else {
            http_response_code(503); // Service Unavailable (database error)
            echo json_encode(array("error" => "Unable to delete user. Database error.", "details" => $stmt->errorInfo()));
        }
    }

} catch (PDOException $e) {
    http_response_code(503); // Service Unavailable
    echo json_encode(array("error" => "Database connection error: " . $e->getMessage()));
} catch (Exception $e) {
    http_response_code(500); // Internal Server Error
    echo json_encode(array("error" => "An unexpected error occurred: " . $e->getMessage()));
} finally {
    $pdo = null; // Close connection
}
?>
