<?php
header("Access-Control-Allow-Origin: *"); // Allow requests from any origin (adjust for production)
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, OPTIONS"); // Allow POST and OPTIONS methods
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request (pre-flight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../../db_connect.php'; // Adjust path as needed

// Get posted data
$data = json_decode(file_get_contents("php://input"));

// Basic validation
if (
    !isset($data->description) || empty($data->description) ||
    !isset($data->severity) || empty($data->severity) ||
    !isset($data->status) || empty($data->status)
) {
    http_response_code(400); // Bad Request
    echo json_encode(array("error" => "Incomplete or invalid defect data provided."));
    exit();
}

// Sanitize input (basic example)
$description = htmlspecialchars(strip_tags($data->description));
$severity = htmlspecialchars(strip_tags($data->severity));
$status = htmlspecialchars(strip_tags($data->status));
$assetId = isset($data->assetId) ? htmlspecialchars(strip_tags($data->assetId)) : null;
$inspectionId = isset($data->inspectionId) ? htmlspecialchars(strip_tags($data->inspectionId)) : null;

// Validate severity and status against allowed enum values (important!)
$allowed_severities = ['minor', 'major', 'critical'];
$allowed_statuses = ['open', 'in_progress', 'resolved'];

if (!in_array($severity, $allowed_severities)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid severity specified."));
    exit();
}

if (!in_array($status, $allowed_statuses)) {
    http_response_code(400);
    echo json_encode(array("error" => "Invalid status specified."));
    exit();
}

try {
    // $pdo is created in db_connect.php
    if (!isset($pdo) || !$pdo instanceof PDO) {
        // This condition might be hit if db_connect.php failed silently
        throw new Exception("Database connection object not available.");
    }

    // Prepare insert statement
    $sql = "INSERT INTO defects (description, severity, status, asset_id, inspection_id, created_at, updated_at) VALUES (:description, :severity, :status, :assetId, :inspectionId, NOW(), NOW())";
    $stmt = $pdo->prepare($sql);

    // Bind parameters
    $stmt->bindParam(':description', $description);
    $stmt->bindParam(':severity', $severity);
    $stmt->bindParam(':status', $status);
    $stmt->bindParam(':assetId', $assetId);
    $stmt->bindParam(':inspectionId', $inspectionId);

    // Execute the query
    if ($stmt->execute()) {
        http_response_code(201); // Created
        echo json_encode(array("message" => "Defect registered successfully."));
    } else {
        http_response_code(503); // Service Unavailable (database error)
        echo json_encode(array("error" => "Unable to register defect. Database error."));
    }

} catch (PDOException $e) {
    http_response_code(503); // Service Unavailable
    echo json_encode(array("error" => "Database connection error: " . $e->getMessage()));
} catch (Exception $e) {
    http_response_code(500); // Internal Server Error
    echo json_encode(array("error" => "An unexpected error occurred: " . $e->getMessage()));
} finally {
    $pdo = null; // Close connection
}
?>
