<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Database connection
require_once 'db_connect.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

// Check for token in headers - disabled for testing
// $headers = getallheaders();
// $token = isset($headers['Authorization']) ? $headers['Authorization'] : '';

// Verify token (simplified for demo)
// In a production environment, implement proper JWT validation
// if (empty($token)) {
//     echo json_encode(['error' => 'Authentication required']);
//     exit;
// }

try {
    $pdo = new PDO("mysql:host=$host;dbname=$database", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    echo json_encode(['error' => 'Database connection failed: ' . $e->getMessage()]);
    exit;
}

if ($method === 'GET') {
    try {
        // Check if role filter is provided
        $role = isset($_GET['role']) ? $_GET['role'] : null;

        // Base query
        $sql = "SELECT
                    id,
                    user_identifier,
                    name,
                    email,
                    role,
                    status,
                    created_at
                FROM
                    users
                WHERE
                    status = 'active'";

        // Add role filter if provided
        if ($role) {
            $sql .= " AND role = :role";
        }

        // Add ordering
        $sql .= " ORDER BY name ASC";

        $stmt = $pdo->prepare($sql);

        // Bind role parameter if provided
        if ($role) {
            $stmt->bindParam(':role', $role, PDO::PARAM_STR);
        }

        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode($users);
        } else {
            // If no users found, return empty array
            echo json_encode([]);
        }
    } catch (PDOException $e) {
        echo json_encode(['error' => 'Failed to fetch users: ' . $e->getMessage()]);
    }
} elseif ($method === 'POST') {
    try {
        // Get POST data
        $data = json_decode(file_get_contents("php://input"), true);

        // Validate required fields
        $requiredFields = ['name', 'email', 'password', 'role'];
        foreach ($requiredFields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                echo json_encode(['error' => "Missing required field: $field"]);
                exit;
            }
        }

        // Check if email already exists
        $checkSql = "SELECT id FROM users WHERE email = ?";
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->execute([$data['email']]);

        if ($checkStmt->rowCount() > 0) {
            echo json_encode(['error' => 'Email already exists']);
            exit;
        }

        // Generate a unique user identifier
        $userIdentifier = 'U' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);

        // Hash password
        $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);

        // Insert new user
        $sql = "INSERT INTO users
                (user_identifier, name, email, password, role, status, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?)";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $userIdentifier,
            $data['name'],
            $data['email'],
            $hashedPassword,
            $data['role'],
            $data['status'] ?? 'active',
            $data['created_by'] ?? null
        ]);

        echo json_encode([
            'success' => true,
            'message' => 'User created successfully',
            'id' => $pdo->lastInsertId()
        ]);
    } catch (PDOException $e) {
        echo json_encode(['error' => 'Failed to create user: ' . $e->getMessage()]);
    }
} else {
    echo json_encode(['error' => 'Method not allowed']);
}
?>
